/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.cxf.transport.http.policy;

import org.apache.cxf.transport.http.policy.impl.ClientPolicyCalculator;
import org.apache.cxf.transports.http.configuration.HTTPClientPolicy;
import org.junit.Assert;
import org.junit.Test;

public class ClientPolicyCalculatorTest extends Assert {
    @Test
    public void testCompatibleClientPolicies() {
        ClientPolicyCalculator calc = new ClientPolicyCalculator();
        HTTPClientPolicy p1 = new HTTPClientPolicy();
        assertTrue("Policy is not compatible with itself.", calc.compatible(p1, p1));
        HTTPClientPolicy p2 = new HTTPClientPolicy();
        assertTrue("Policies are not compatible.", calc.compatible(p1, p2));
        p1.setBrowserType("browser");
        assertTrue("Policies are not compatible.", calc.compatible(p1, p2));
        p1.setBrowserType(null);
        p1.setConnectionTimeout(10000);
        assertTrue("Policies are not compatible.", calc.compatible(p1, p2));
        p1.setAllowChunking(false);
        assertTrue("Policies are compatible.", !calc.compatible(p1, p2));
        p2.setAllowChunking(false);
        assertTrue("Policies are compatible.", calc.compatible(p1, p2));
    }

    @Test
    public void testIntersectClientPolicies() {
        ClientPolicyCalculator calc = new ClientPolicyCalculator();
        HTTPClientPolicy p1 = new HTTPClientPolicy();
        HTTPClientPolicy p2 = new HTTPClientPolicy();
        HTTPClientPolicy p = null;

        p1.setBrowserType("browser");
        p = calc.intersect(p1, p2);
        assertEquals("browser", p.getBrowserType());
        p1.setBrowserType(null);
        p1.setConnectionTimeout(10000L);
        p = calc.intersect(p1, p2);
        assertEquals(10000L, p.getConnectionTimeout());
        p1.setAllowChunking(false);
        p2.setAllowChunking(false);
        p = calc.intersect(p1, p2);
        assertTrue(!p.isAllowChunking());
    }
    
    @Test
    public void testEqualClientPolicies() {
        ClientPolicyCalculator calc = new ClientPolicyCalculator();
        HTTPClientPolicy p1 = new HTTPClientPolicy();
        assertTrue(calc.equals(p1, p1));
        HTTPClientPolicy p2 = new HTTPClientPolicy();        
        assertTrue(calc.equals(p1, p2));
        p1.setDecoupledEndpoint("http://localhost:8080/decoupled");
        assertTrue(!calc.equals(p1, p2));
        p2.setDecoupledEndpoint("http://localhost:8080/decoupled");
        assertTrue(calc.equals(p1, p2));
        p1.setReceiveTimeout(10000L);
        assertTrue(!calc.equals(p1, p2));
    }

}
