/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.http.endpoints;

import java.io.IOException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.activation.DataHandler;
import javax.jbi.management.DeploymentException;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.InOptionalOut;
import javax.jbi.messaging.InOut;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.NormalizedMessage;
import javax.jbi.servicedesc.ServiceEndpoint;
import javax.xml.namespace.QName;

import org.apache.servicemix.JbiConstants;
import org.apache.servicemix.common.DefaultComponent;
import org.apache.servicemix.common.ServiceUnit;
import org.apache.servicemix.common.endpoints.ProviderEndpoint;
import org.apache.servicemix.http.HttpComponent;
import org.apache.servicemix.http.HttpEndpointType;
import org.apache.servicemix.http.jetty.SmxHttpExchange;
import org.apache.servicemix.http.tools.AttachmentProcessorUtil;
import org.mortbay.jetty.client.HttpClient;

/**
 * @author gnodet
 * @org.apache.xbean.XBean element="provider"
 * @since 3.2
 */
public class HttpProviderEndpoint extends ProviderEndpoint implements HttpEndpointType {

    /**
     * If true, the accept-encoding http header will be set to gzip and the response will be un-gzipped.
     */
    protected boolean expectGzippedResponse;

    /**
     * If true, the request content will be gzipped and sent over the wire. The content-encoding http header will
     * also be set to gzip.
     */
    protected boolean gzipRequest;

    //private SslParameters ssl;
    //private BasicAuthCredentials basicAuthentication;
    private HttpProviderMarshaler marshaler;
    private String locationURI;
    private int clientSoTimeout = 10000;
    private HttpClient jettyClient;

    /**
     * How will the endpoint support attachments (for provider endpoint only). Possible values:
     * - true - message attachment will always be included in the http request
     * - ignore - message attachment will be silently dropped or transferred to the out message
     * - warn - message attachment will be dropped or transferred to the out message, but a warning message will be logged
     * - fail - an exception will be thrown if an attachment is detected
     */
    private String attachmentSupport = AttachmentProcessorUtil.ATTACHMENT_SUPPORT_TRUE;

    // Temporary cache that stores nms attachments that will be transferred to the out message when
    // attachment support is set to "warn" or "ignore"
    private Map<String, Map<String, DataHandler>> attachmentTransfer = new ConcurrentHashMap<String, Map<String, DataHandler>>();

    public HttpProviderEndpoint() {
        super();
    }

    public HttpProviderEndpoint(DefaultComponent component, ServiceEndpoint endpoint) {
        super(component, endpoint);
    }

    public HttpProviderEndpoint(ServiceUnit serviceUnit, QName service, String endpoint) {
        super(serviceUnit, service, endpoint);
    }

    public String getLocationURI() {
        return locationURI;
    }

    public void setLocationURI(String locationURI) {
        this.locationURI = locationURI;
    }

    /*
    public BasicAuthCredentials getBasicAuthentication() {
        return basicAuthentication;
    }

    public void setBasicAuthentication(BasicAuthCredentials basicAuthentication) {
        this.basicAuthentication = basicAuthentication;
    }


    public SslParameters getSsl() {
        return ssl;
    }

    public void setSsl(SslParameters ssl) {
        this.ssl = ssl;
    }
    */

    /**
     * @return the marshaler
     */
    public HttpProviderMarshaler getMarshaler() {
        return marshaler;
    }

    /**
     * @param marshaler the marshaler to set
     */
    public void setMarshaler(HttpProviderMarshaler marshaler) {
        this.marshaler = marshaler;
    }

    public void process(MessageExchange exchange) throws Exception {
        if (exchange.getStatus() == ExchangeStatus.ACTIVE) {
            NormalizedMessage nm = exchange.getMessage("in");
            if (nm == null) {
                throw new IllegalStateException("Exchange has no input message");
            }

            Map<String, DataHandler> attachments = AttachmentProcessorUtil.preProcessAttachments(attachmentSupport, exchange, nm);
            if (attachments != null) {
                attachmentTransfer.put(exchange.getExchangeId(), attachments);
            }

            SmxHttpExchange httpEx = new Exchange(exchange);
            marshaler.createRequest(exchange, nm, httpEx);
            getConnectionPool().send(httpEx);
        }
    }

    protected void handle(SmxHttpExchange httpExchange, MessageExchange exchange) throws IOException {
        try {
            marshaler.handleResponse(exchange, httpExchange);

            // Transfer cached attachment to out message
            if (exchange instanceof InOut || exchange instanceof InOptionalOut) {
                Map<String, DataHandler> attachments = attachmentTransfer.get(exchange.getExchangeId());
                if (attachments != null) {
                    AttachmentProcessorUtil.transferAttachments(attachments, exchange.getMessage("out"));
                }
            }

            boolean txSync = exchange.getStatus() == ExchangeStatus.ACTIVE
                && exchange.isTransacted()
                && Boolean.TRUE.equals(exchange.getProperty(JbiConstants.SEND_SYNC));
            if (txSync) {
                sendSync(exchange);
            } else {
                send(exchange);
            }
        } catch (Exception e) {
            throw (IOException) new IOException(e.getMessage()).initCause(e);
        }
    }
    
    protected void handleConnectionFailed(Throwable throwable, MessageExchange exchange) {
        try {
            Exception e;
            if (throwable instanceof Exception) {
                e = (Exception) throwable;
            } else {
                e = new Exception(throwable);
            }
            exchange.setError(e);
            send(exchange);
        } catch (Exception e) {
            logger.warn("Unable to send back exchange in error", e);
        }
    }

    protected org.mortbay.jetty.client.HttpClient getConnectionPool() throws Exception {
        if (jettyClient == null) {
            HttpComponent comp = (HttpComponent) getServiceUnit().getComponent();
            if (comp.getConfiguration().isJettyClientPerProvider()) {
                jettyClient = comp.getNewJettyClient(comp);
            } else {
                //return shared client
                jettyClient = comp.getConnectionPool();
            }
            jettyClient.setSoTimeout(getClientSoTimeout());
        }
        return jettyClient;
    }

    protected class Exchange extends SmxHttpExchange {

        MessageExchange jbiExchange;

        public Exchange(MessageExchange jbiExchange) {
            this.jbiExchange = jbiExchange;
        }

        protected void onResponseComplete() throws IOException {
            handle(this, jbiExchange);
        }
        protected void onException(Throwable throwable) {
            throwable.printStackTrace();
            throw new RuntimeException(throwable);
        }
        protected void onConnectionFailed(Throwable throwable) {
            handleConnectionFailed(throwable, jbiExchange);
        }
    }

    public int getClientSoTimeout() {
        return clientSoTimeout;
    }

    public void setClientSoTimeout(int clientTimeout) {
        this.clientSoTimeout = clientTimeout;
    }

    public String getAttachmentSupport() {
        return attachmentSupport;
    }

    public void setAttachmentSupport(String attachmentSupport) {
        this.attachmentSupport = attachmentSupport;
    }

    public boolean isGzipRequest() {
        return gzipRequest;
    }

    public void setGzipRequest(boolean gzipRequest) {
        this.gzipRequest = gzipRequest;
    }

    public boolean isExpectGzippedResponse() {
        return expectGzippedResponse;
    }

    public void setExpectGzippedResponse(boolean expectGzippedResponse) {
        this.expectGzippedResponse = expectGzippedResponse;
    }

    public void validate() throws DeploymentException {
        super.validate();
        if (marshaler == null) {
            marshaler = new DefaultHttpProviderMarshaler();
        }
        if (marshaler instanceof DefaultHttpProviderMarshaler) {
            ((DefaultHttpProviderMarshaler) marshaler).setLocationURI(locationURI);
            if (isGzipRequest()) {
                ((DefaultHttpProviderMarshaler) marshaler).setContentEncoding("gzip");
            }
            if (isExpectGzippedResponse()) {
                ((DefaultHttpProviderMarshaler) marshaler).setAcceptEncoding("gzip");
            }
        }
    }
}
