/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.osworkflow;

import javax.jbi.JBIException;
import javax.jbi.component.Bootstrap;
import javax.jbi.component.InstallationContext;
import javax.management.MBeanServer;
import javax.management.ObjectName;

/**
 * Bootstrap class. This class is useful to perform tasks at installation /
 * uninstallation time
 * 
 * @author lhe
 */
public class OSWorkflowBootstrap implements Bootstrap {
    protected InstallationContext context;

    protected ObjectName mbeanName;

    /**
     * creates a new bootstrap
     */
    public OSWorkflowBootstrap() {
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.component.Bootstrap#getExtensionMBeanName()
     */
    public ObjectName getExtensionMBeanName() {
        return mbeanName;
    }

    /**
     * returns the extension mbean
     * 
     * @return the object
     */
    protected Object getExtensionMBean() throws Exception {
        return null;
    }

    /**
     * creates the extension mbean object name
     * 
     * @return the object name
     * @throws Exception
     */
    protected ObjectName createExtensionMBeanName() throws Exception {
        return this.context.getContext().getMBeanNames()
                .createCustomComponentMBeanName("bootstrap");
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.component.Bootstrap#init(javax.jbi.component.InstallationContext)
     */
    public void init(InstallationContext installContext) throws JBIException {
        try {
            this.context = installContext;
            doInit();
        } catch (JBIException e) {
            throw e;
        } catch (Exception e) {
            throw new JBIException("Error calling init", e);
        }
    }

    /**
     * 
     * @throws Exception
     */
    protected void doInit() throws Exception {
        Object mbean = getExtensionMBean();
        if (mbean != null) {
            this.mbeanName = createExtensionMBeanName();
            MBeanServer server = this.context.getContext().getMBeanServer();
            if (server == null) {
                throw new JBIException("null mBeanServer");
            }
            if (server.isRegistered(this.mbeanName)) {
                server.unregisterMBean(this.mbeanName);
            }
            server.registerMBean(mbean, this.mbeanName);
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.component.Bootstrap#cleanUp()
     */
    public void cleanUp() throws JBIException {
        try {
            doCleanUp();
        } catch (JBIException e) {
            throw e;
        } catch (Exception e) {
            throw new JBIException("Error calling cleanUp", e);
        }
    }

    /**
     * clean up
     * 
     * @throws Exception
     */
    protected void doCleanUp() throws Exception {
        if (this.mbeanName != null) {
            MBeanServer server = this.context.getContext().getMBeanServer();
            if (server == null) {
                throw new JBIException("null mBeanServer");
            }
            if (server.isRegistered(this.mbeanName)) {
                server.unregisterMBean(this.mbeanName);
            }
        }
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.component.Bootstrap#onInstall()
     */
    public void onInstall() throws JBIException {
        try {
            doOnInstall();
        } catch (JBIException e) {
            throw e;
        } catch (Exception e) {
            throw new JBIException("Error calling onInstall", e);
        }
    }

    /**
     * called upon installation
     * 
     * @throws Exception
     */
    protected void doOnInstall() throws Exception {
    }

    /*
     * (non-Javadoc)
     * 
     * @see javax.jbi.component.Bootstrap#onUninstall()
     */
    public void onUninstall() throws JBIException {
        try {
            doOnUninstall();
        } catch (JBIException e) {
            throw e;
        } catch (Exception e) {
            throw new JBIException("Error calling onUninstall", e);
        }
    }

    /**
     * called upon uninstallation
     * 
     * @throws Exception
     */
    protected void doOnUninstall() throws Exception {
    }
}
