/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.http.endpoints;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.URI;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import javax.jbi.component.ComponentContext;
import javax.jbi.messaging.Fault;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.NormalizedMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.transform.stream.StreamSource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.servicemix.jbi.jaxp.StAXSourceTransformer;
import org.apache.servicemix.jbi.jaxp.XMLStreamHelper;
import org.apache.servicemix.jbi.messaging.MessageExchangeSupport;
import org.mortbay.jetty.HttpHeaders;

/**
 * The default consumer marshaler used for non-soap consumer endpoints.
 *
 * @author gnodet
 * @since 3.2
 */
public class DefaultHttpConsumerMarshaler implements HttpConsumerMarshaler {

    private static Log log = LogFactory.getLog(DefaultHttpConsumerMarshaler.class);
    private StAXSourceTransformer transformer = new StAXSourceTransformer();
    private URI defaultMep;


    public DefaultHttpConsumerMarshaler() {
        this(MessageExchangeSupport.IN_OUT);
    }

    public DefaultHttpConsumerMarshaler(URI defaultMep) {
        this.defaultMep = defaultMep;
    }

    public URI getDefaultMep() {
        return defaultMep;
    }

    public void setDefaultMep(URI defaultMep) {
        this.defaultMep = defaultMep;
    }

    public MessageExchange createExchange(HttpServletRequest request, ComponentContext context) throws Exception {
        MessageExchange me = context.getDeliveryChannel().createExchangeFactory().createExchange(getDefaultMep());
        NormalizedMessage in = me.createMessage();
        in.setContent(new StreamSource(getRequestEncodingStream(request.getHeader(HttpHeaders.CONTENT_ENCODING),
            request.getInputStream())));
        me.setMessage(in, "in");
        return me;
    }

    public void sendOut(MessageExchange exchange, NormalizedMessage outMsg, HttpServletRequest request,
                        HttpServletResponse response) throws Exception {
        addResponseHeaders(exchange, request, response);
        XMLStreamReader reader = transformer.toXMLStreamReader(outMsg.getContent());
        OutputStream encodingStream = getResponseEncodingStream(request.getHeader(HttpHeaders.ACCEPT_ENCODING),
            response.getOutputStream());
        XMLStreamWriter writer = transformer.getOutputFactory().createXMLStreamWriter(encodingStream);
        writer.writeStartDocument();
        XMLStreamHelper.copy(reader, writer);
        writer.writeEndDocument();
        writer.flush();
        encodingStream.flush();
        writer.close();
        encodingStream.close();

        response.setStatus(HttpServletResponse.SC_OK);
    }

    public void sendFault(MessageExchange exchange, Fault fault, HttpServletRequest request,
                          HttpServletResponse response) throws Exception {
        addResponseHeaders(exchange, request, response);
        response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        XMLStreamReader reader = transformer.toXMLStreamReader(fault.getContent());

        OutputStream encodingStream = getResponseEncodingStream(request.getHeader(HttpHeaders.ACCEPT_ENCODING),
            response.getOutputStream());
        XMLStreamWriter writer = transformer.getOutputFactory().createXMLStreamWriter(encodingStream);
        XMLStreamHelper.copy(reader, writer);
        writer.flush();
        encodingStream.flush();
        writer.close();
        encodingStream.close();
    }

    public void sendError(MessageExchange exchange, Exception error, HttpServletRequest request,
                          HttpServletResponse response) throws Exception {
        addResponseHeaders(exchange, request, response);
        response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
        OutputStream encodingStream = getResponseEncodingStream(request.getHeader(HttpHeaders.ACCEPT_ENCODING),
            response.getOutputStream());
        XMLStreamWriter writer = transformer.getOutputFactory().createXMLStreamWriter(encodingStream);
        writer.writeStartDocument();
        writer.writeStartElement("error");
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        error.printStackTrace(pw);
        pw.close();
        writer.writeCData(sw.toString());
        writer.writeEndElement();
        writer.writeEndDocument();
        writer.flush();
        encodingStream.flush();
        writer.close();
        encodingStream.close();
    }

    public void sendAccepted(MessageExchange exchange, HttpServletRequest request, HttpServletResponse response)
        throws Exception {
        response.setStatus(HttpServletResponse.SC_ACCEPTED);
    }

    protected InputStream getRequestEncodingStream(String contentEncoding, InputStream dataStream) throws IOException {
        if (contentEncoding != null && contentEncoding.toLowerCase().indexOf("gzip") >= 0) {
            log.debug("Using GZIP request content encoding.");
            return new GZIPInputStream(new BufferedInputStream(dataStream));
        } else {
            log.debug("Using default request content encoding.");
            return new BufferedInputStream(dataStream);
        }
    }

    protected OutputStream getResponseEncodingStream(String acceptEncoding, OutputStream dataStream) throws IOException {
        if (acceptEncoding != null && acceptEncoding.toLowerCase().indexOf("gzip") >= 0) {
            log.debug("Using GZIP response content encoding.");
            return new GZIPOutputStream(new BufferedOutputStream(dataStream));
        } else {
            log.debug("Using default response content encoding.");
            return new BufferedOutputStream(dataStream);
        }
    }

    protected void addResponseHeaders(MessageExchange exchange, HttpServletRequest request, HttpServletResponse response) {
        String acceptEncoding = request.getHeader(HttpHeaders.ACCEPT_ENCODING);
        if (acceptEncoding != null && acceptEncoding.toLowerCase().indexOf("gzip") >= 0) {
            response.setHeader(HttpHeaders.CONTENT_ENCODING, "gzip");
        }
    }
}
