/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.http.tools;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.activation.DataHandler;
import javax.jbi.messaging.InOptionalOut;
import javax.jbi.messaging.InOut;
import javax.jbi.messaging.MessageExchange;
import javax.jbi.messaging.MessagingException;
import javax.jbi.messaging.NormalizedMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public final class AttachmentProcessorUtil {

    public static final String ATTACHMENT_SUPPORT_TRUE = "true";
    public static final String ATTACHMENT_SUPPORT_IGNORE = "ignore";
    public static final String ATTACHMENT_SUPPORT_WARN = "warn";
    public static final String ATTACHMENT_SUPPORT_FAIL = "fail";

    private static Log log = LogFactory.getLog(AttachmentProcessorUtil.class);

    private AttachmentProcessorUtil() {
    }

    @SuppressWarnings("unchecked")
    public static Map<String, DataHandler> preProcessAttachments(String attSupport,
                    MessageExchange exchange, NormalizedMessage inMsg) throws MessagingException {

        Map<String, DataHandler> attachmentTransfer = null;
        // If there are attachments, check to see if they are supported or not
        if (inMsg.getAttachmentNames().size() > 0) {
            if (ATTACHMENT_SUPPORT_TRUE.equalsIgnoreCase(attSupport)) {
                return null; // Do nothing if attachment is supported
            } else if (ATTACHMENT_SUPPORT_FAIL.equalsIgnoreCase(attSupport)) {
                // throw an exception if not supported and should fail
                throw new IllegalStateException("Message attachment not supported in http request, "
                    + "but normalized message contains one or more attachments.");
            } else {
                if (ATTACHMENT_SUPPORT_WARN.equalsIgnoreCase(attSupport)) {
                    // display a warn message if not supported and attachments should be dropped
                    log.warn("Message attachment not supported in http request. One or more attachments are not being sent.");
                }

                if (exchange instanceof InOut || exchange instanceof InOptionalOut) {
                    // Cache attachments for transfer later
                    attachmentTransfer = new HashMap<String, DataHandler>();
                    Set<String> attachmentNames = inMsg.getAttachmentNames();
                    for (String attachmentName : attachmentNames) {
                        attachmentTransfer.put(attachmentName, inMsg.getAttachment(attachmentName));
                        inMsg.removeAttachment(attachmentName);
                    }

                } else {
                    // Drop attachments
                    Set<String> attachmentNames = inMsg.getAttachmentNames();
                    for (String attachmentName : attachmentNames) {
                        inMsg.removeAttachment(attachmentName);
                    }
                }
            }
        }
        return attachmentTransfer;
    }

    public static void transferAttachments(Map<String, DataHandler> attachmentTransfer,
                                           NormalizedMessage dest) throws MessagingException {
        for (Map.Entry<String, DataHandler> att : attachmentTransfer.entrySet()) {
            dest.addAttachment(att.getKey(), att.getValue());
        }
    }
}
