/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.servicemix.http;

import javax.activation.DataHandler;
import javax.jbi.messaging.ExchangeStatus;
import javax.jbi.messaging.InOut;
import javax.xml.namespace.QName;

import junit.framework.TestCase;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.servicemix.client.DefaultServiceMixClient;
import org.apache.servicemix.client.ServiceMixClient;
import org.apache.servicemix.components.util.EchoComponent;
import org.apache.servicemix.http.endpoints.HttpConsumerEndpoint;
import org.apache.servicemix.http.endpoints.HttpProviderEndpoint;
import org.apache.servicemix.http.tools.AttachmentProcessorUtil;
import org.apache.servicemix.jbi.container.JBIContainer;
import org.apache.servicemix.jbi.jaxp.SourceTransformer;
import org.apache.servicemix.jbi.jaxp.StringSource;
import org.apache.servicemix.jbi.util.ByteArrayDataSource;

public class HttpAttachmentSupportTest extends TestCase {

    private static Log logger = LogFactory.getLog(HttpAttachmentSupportTest.class);

    protected JBIContainer container;

    protected void setUp() throws Exception {
        container = new JBIContainer();
        container.setUseMBeanServer(false);
        container.setCreateMBeanServer(false);
        container.setEmbedded(true);
        container.init();
    }

    protected void tearDown() throws Exception {
        if (container != null) {
            container.shutDown();
            Thread.sleep(1000);
        }
    }

    public void testAttachmentSupportTrue() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpEndpoint consumer = new HttpEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");
        consumer.setRoleAsString("consumer");

        HttpEndpoint provider = new HttpEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setRoleAsString("provider");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_TRUE);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                throw me.getError();
            } else {
                fail("Received ERROR status");
            }
        } else {
            assertNotNull(me.getOutMessage().getAttachment("att1"));
            logger.info(new SourceTransformer().toString(me.getOutMessage().getContent()));
        }
        client.done(me);
    }

    public void testAttachmentSupportFail() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpEndpoint consumer = new HttpEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");
        consumer.setRoleAsString("consumer");

        HttpEndpoint provider = new HttpEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setRoleAsString("provider");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_FAIL);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                // success
            } else {
                fail("Received ERROR status");
            }
        } else {
            fail("Should have received error that attachments is not supported.");
        }
    }

    public void testAttachmentSupportWarn() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpEndpoint consumer = new HttpEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");
        consumer.setRoleAsString("consumer");

        HttpEndpoint provider = new HttpEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setRoleAsString("provider");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_WARN);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                throw me.getError();
            } else {
                fail("Received ERROR status");
            }
        } else {
            // Check that attachment has been transferred to out
            assertNotNull(me.getOutMessage().getAttachment("att1"));
            logger.info(new SourceTransformer().toString(me.getOutMessage().getContent()));
        }
        client.done(me);
    }

    // Ignore this test case as its failing even without the attachment feature implemented
    public void xtestAttachmentSupportTrueNewHttpEndpoint() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpConsumerEndpoint consumer = new HttpConsumerEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");

        HttpProviderEndpoint provider = new HttpProviderEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_TRUE);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                throw me.getError();
            } else {
                fail("Received ERROR status");
            }
        } else {
            assertNotNull(me.getOutMessage().getAttachment("att1"));
            logger.info(new SourceTransformer().toString(me.getOutMessage().getContent()));
        }
        client.done(me);
    }

    public void testAttachmentSupportFailNewHttpEndpoint() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpConsumerEndpoint consumer = new HttpConsumerEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");

        HttpProviderEndpoint provider = new HttpProviderEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_FAIL);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                // success
            } else {
                fail("Received ERROR status");
            }
        } else {
            fail("Should have received error that attachments is not supported.");
        }
    }

    public void testAttachmentSupportWarnNewHttpEndpoint() throws Exception {
        EchoComponent echo = new EchoComponent();
        echo.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        echo.setEndpoint("service");
        container.activateComponent(echo, "echo");

        HttpComponent http = new HttpComponent();

        HttpConsumerEndpoint consumer = new HttpConsumerEndpoint();
        consumer.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        consumer.setEndpoint("consumer");
        consumer.setTargetService(new QName("http://servicemix.apache.org/samples/wsdl-first", "EchoService"));
        consumer.setTargetEndpoint("service");
        consumer.setLocationURI("http://localhost:8192/person/");

        HttpProviderEndpoint provider = new HttpProviderEndpoint();
        provider.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        provider.setEndpoint("provider");
        provider.setLocationURI("http://localhost:8192/person/");
        provider.setAttachmentSupport(AttachmentProcessorUtil.ATTACHMENT_SUPPORT_WARN);

        http.setEndpoints(new HttpEndpointType[] {consumer, provider });
        container.activateComponent(http, "http");
        container.start();

        ServiceMixClient client = new DefaultServiceMixClient(container);
        InOut me = client.createInOutExchange();
        me.setService(new QName("http://servicemix.apache.org/samples/wsdl-first", "PersonService"));
        me.getInMessage().setContent(new StringSource("<echo xmlns='http://test'><echoin0>world</echoin0></echo>"));
        me.getInMessage().addAttachment("att1", new DataHandler(
            new ByteArrayDataSource(new byte[] {0, 1, 2 }, "image/jpg")));
        client.sendSync(me);

        if (me.getStatus() == ExchangeStatus.ERROR) {
            if (me.getFault() != null) {
                fail("Received fault: " + new SourceTransformer().toString(me.getFault().getContent()));
            } else if (me.getError() != null) {
                throw me.getError();
            } else {
                fail("Received ERROR status");
            }
        } else {
            // Check that attachment has been transferred to out
            assertNotNull(me.getOutMessage().getAttachment("att1"));
            logger.info(new SourceTransformer().toString(me.getOutMessage().getContent()));
        }
        client.done(me);
    }
}
