/*
 * Copyright 2014 ArcBees Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.gwtplatform.dispatch.rest.client.interceptor;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.gwtplatform.dispatch.client.interceptor.AbstractInterceptor;
import com.gwtplatform.dispatch.rest.shared.RestAction;
import com.gwtplatform.dispatch.shared.TypedAction;

/**
 * Simple abstract super-class for {@link RestInterceptor} implementations that forces the action class to be passed in
 * as a constructor to the handler.
 */
public abstract class AbstractRestInterceptor extends AbstractInterceptor<RestAction, Object>
        implements RestInterceptor {
    private final List<InterceptorContext> interceptorContexts;

    protected AbstractRestInterceptor(
            InterceptorContext context,
            InterceptorContext... moreContexts) {
        super(RestAction.class);

        List<InterceptorContext> contexts = new ArrayList<InterceptorContext>();
        contexts.add(context);

        if (moreContexts != null) {
            Collections.addAll(contexts, moreContexts);
        }

        interceptorContexts = Collections.unmodifiableList(contexts);
    }

    @Override
    public List<InterceptorContext> getInterceptorContexts() {
        return interceptorContexts;
    }

    @Override
    public boolean canExecute(TypedAction<?> action) {
        if (action instanceof RestAction) {
            InterceptorContext subjectContext = new InterceptorContext.Builder((RestAction<?>) action).build();

            for (InterceptorContext context : interceptorContexts) {
                // Must have at least one supporting context
                if (context.equals(subjectContext)) {
                    return true;
                }
            }
        }
        return false;
    }
}
