package dev.langchain4j.model.chat.response;

import dev.langchain4j.Experimental;
import dev.langchain4j.model.chat.StreamingChatModel;

/**
 * Represents a handler for a {@link StreamingChatModel} response.
 *
 * @see StreamingChatModel
 */
public interface StreamingChatResponseHandler {

    /**
     * Invoked each time the model generates a partial textual response, usually a single token.
     * <p>
     * Please note that some LLM providers do not stream individual tokens, but send responses in batches.
     * In such cases, this callback may receive multiple tokens at once.
     *
     * @param partialResponse A partial textual response, usually a single token.
     */
    void onPartialResponse(String partialResponse);

    /**
     * Invoked each time the model generates a partial thinking/reasoning text, usually a single token.
     * <p>
     * Please note that some LLM providers do not stream individual tokens, but send thinking tokens in batches.
     * In such cases, this callback may receive multiple tokens at once.
     *
     * @param partialThinking A partial thinking text, usually a single token.
     * @since 1.2.0
     */
    @Experimental
    default void onPartialThinking(PartialThinking partialThinking) {
    }

    /**
     * This callback is invoked each time the model generates a partial tool call,
     * which contains a single token of the tool's arguments.
     * It is typically invoked multiple times for a single tool call
     * until {@link #onCompleteToolCall(CompleteToolCall)} is eventually invoked,
     * indicating that the streaming for that tool call is finished.
     * <p>
     * Here's an example of what streaming a single tool call might look like:
     * <pre>
     * 1. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = "{\"")
     * 2. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = "city")
     * 3. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = ""\":\"")
     * 4. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = "Mun")
     * 5. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = "ich")
     * 6. onPartialToolCall(index = 0, id = "call_abc", name = "get_weather", partialArguments = "\"}")
     * 7. onCompleteToolCall(index = 0, id = "call_abc", name = "get_weather", arguments = "{\"city\":\"Munich\"}")
     * </pre>
     * <p>
     * If the model decides to call multiple tools, the index will increment, allowing you to correlate.
     * <p>
     * Please note that not all LLM providers stream tool calls token by token.
     * Some providers (e.g., Bedrock, Google, Mistral, Ollama) return only complete tool calls.
     * In those cases, this callback won't be invoked - only {@link #onCompleteToolCall(CompleteToolCall)}
     * will be called.
     *
     * @param partialToolCall A partial tool call that contains
     *                        the index, tool ID, tool name and partial arguments.
     * @since 1.2.0
     */
    @Experimental
    default void onPartialToolCall(PartialToolCall partialToolCall) {
    }

    /**
     * Invoked when the model has finished streaming a single tool call.
     *
     * @param completeToolCall A complete tool call that contains
     *                         the index, tool ID, tool name, and fully assembled arguments.
     * @since 1.2.0
     */
    @Experimental
    default void onCompleteToolCall(CompleteToolCall completeToolCall) {
    }

    /**
     * Invoked when the model has finished streaming a response.
     *
     * @param completeResponse The complete response generated by the model,
     *                         containing all assembled partial text and tool calls.
     */
    void onCompleteResponse(ChatResponse completeResponse);

    /**
     * This method is invoked when an error occurs during streaming.
     *
     * @param error The error that occurred
     */
    void onError(Throwable error);
}
