/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client;

import java.util.LinkedList;
import java.util.List;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

/**
 * This abstract class should serve as a base class for
 * multi field validators. These are validators that need to
 * consider more than field to do their validations.
 * 
 * For example given two password fields that neeed to be checked
 * if the input is the same (see <code>MultiStringsEqualsValidator</code>).
 * 
 * @author Anatol Mayen
 *
 * @param <T>
 */
public abstract class MultiFieldValidator<T> extends Validator<T> {
	
	private List<TextBoxBase> textBoxes = null;
	private List<SuggestBox> suggestBoxes = null;


	public MultiFieldValidator(boolean preventsPropagationOfValidationChain, TextBoxBase... texts) {
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		textBoxes = new LinkedList<TextBoxBase>();
		if(texts.length < 2)
			throw new IllegalArgumentException("No use in using MultiFieldValidator if you are using less than two fields");
		for(int i=0; i< texts.length; i++) {
			if(texts[i] == null)
				throw new IllegalArgumentException("Null values are not allowed");
			textBoxes.add(0, texts[i]);
		}
	}
	
	public MultiFieldValidator(boolean preventsPropagationOfValidationChain, SuggestBox... texts) {
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		suggestBoxes = new LinkedList<SuggestBox>();
		if(texts.length < 2)
			throw new IllegalArgumentException("No use in using MultiFieldValidator if you are using less than two fields");
		for(int i=0; i< texts.length; i++) {
			if(texts[i] == null)
				throw new IllegalArgumentException("Null values are not allowed");
			suggestBoxes.add(0, texts[i]);
		}
		
	}
	
	/**
	 * Returns a list of the input Strings in the order in which
	 * the widgets were added to this MultiFieldValidator.
	 * 
	 * This method transparently handles if the values come from
	 * a simple textBox or from a suggestBox
	 * 
	 * @return A list with all the values
	 */
	protected List<String> getInputValues() {
		List<String> values = new LinkedList<String>();
		if(isInTextBoxMode()) {
			for(TextBoxBase b : textBoxes)
				values.add(b.getText());
		}else {
			for(SuggestBox b : suggestBoxes)
				values.add(b.getText());
		}
		
		return values;
	}
	

	protected List<TextBoxBase> getTextBoxes() {
		return textBoxes;
	}
	
	protected List<SuggestBox> getSuggestBoxes() {
		return suggestBoxes;
	}
	
	protected boolean isInTextBoxMode() {
		return textBoxes != null;
	}

}
