/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.i18n.utils;

import com.google.gwt.core.client.GWT;


/**
 * The message interpolator allows you to merge arguments
 * into specified string that contain placeholders.
 * 
 * It can be used to assist in the creation of custom validation
 * messages in case they need to be parameterized.
 * 
 * You can do the following.
 * 
 * Overwrite the ValidationMessages.getCustomMessage(String msgKey, Object... args)
 * method with something like the following:
 * 
 * @Override
 * public void getCustomMessage(String msgKey, Object... args) {
 * 
 * 	    String localizedMessage = "Length must be exactly {0}. It was {1}.";
 * 	    
 *      String interpolatedMessage = MessageInterpolator.merge(localizedMessage, 5, 8);
 * 	    //interpolatedMessage now contains: "Length must be exactly 5. It was 8."
 * 	    return interpolatedMessage;
 * }
 * 
 * 
 * @author Anatol Mayen
 *
 */
public class MessageInterpolator {

	
	public static String merge(String message, Object... args) {
		return merge(message, false, args);
	}
	
	public static String merge(String message, boolean silentOnError, Object... args) {
		if(message==null)
			throw new IllegalArgumentException("Message must not be null");
		if(message.trim().length()==0)
			throw new IllegalArgumentException("Message must be given");
		
		//No arguments given, no interpolation needed
		if(args == null || args.length < 1)
			return message;
		
		StringBuffer interpolatedMessage = new StringBuffer();
		int fromIndex = 0;
		int nextIndex = -1;
		while((nextIndex = message.indexOf("{", fromIndex)) != -1) {
			interpolatedMessage.append(message.substring(fromIndex, nextIndex));
			fromIndex = nextIndex+1; //skip everything read till now on the next run
			if(nextIndex > 0) {
				if(message.charAt(nextIndex-1) == '\\') {
					interpolatedMessage.append(message.charAt(nextIndex));
					continue; // We found an escaped bracket '\{', this means we don't interpolate it
				}/*else
					interpolatedMessage.append("\\");*/
				
			}
			
			int closeIndex = message.indexOf('}',nextIndex+1);
			fromIndex = closeIndex+1; //set marker even further
			if(closeIndex == -1) {
				if(!silentOnError)
					throw new RuntimeException("Parameter indicator was not closed properly (Index: "+nextIndex+")! If you want the bracket to be shown in your interpolated message prefix it with '\\'.");
				
				continue; //Interpolation indicator not correctly closed
			}
			
			String argNumber = message.substring(nextIndex+1, closeIndex);
			if(argNumber == null || argNumber.trim().length()==0) {
				if(!silentOnError)
					throw new RuntimeException("No index given for parameter indicator. Message is: "+message);
				interpolatedMessage.append(args[0].toString()); //No number given, append the first argument
			}
				
			else {
				try {
					int argIndex = Integer.valueOf(argNumber);
					if(argIndex >= args.length) {
						if(!silentOnError)
							throw new RuntimeException("Wrong argument index given: "+argIndex+" where "+(args.length-1)+" is the max index! Message was: "+message);
						else
							continue;
					}
						
					interpolatedMessage.append(args[argIndex]);
				}catch(NumberFormatException ex) {
					if(!silentOnError)
						throw new RuntimeException("Parameter index in parameter indicator is not an integer: "+argNumber);
					interpolatedMessage.append(args[0].toString()); //Wrong number given, append the first argument
				}
			}
		}
		
		if(fromIndex < message.length()) //Add remaining string
			interpolatedMessage.append(message.substring(fromIndex));
		
		return interpolatedMessage.toString();
	}
	
	public static void main(String args[]) {
		String message = "One {1} substitution. {0}!";
		String message2 = "One  substitution. {0}!";
		
		System.out.println("Result: "+MessageInterpolator.merge(message, 130, "abc"));
		System.out.println("Result2: "+MessageInterpolator.merge(message2, 130));
	}
	
}
