package eu.maydu.gwt.validation.client.validators.datetime;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBox;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;
import eu.maydu.gwt.validation.client.validators.ValidatorAlgorithmResult;
import eu.maydu.gwt.validation.client.validators.datetime.algorithms.TimeValidatorAlgorithm;


/**
 * This validator validates that a given input is a time of day.
 * 
 * Currently it only validates the european 24 hour format with
 * seconds optionally (This is configurable).
 * 
 * 
 * @author Anatol Mayen
 *
 */
public class TimeValidator extends Validator<TimeValidator> {

	protected TextBox text;
	protected SuggestBox suggest;
	
	private TimeValidatorAlgorithm algorithm;
	
	public TimeValidator(TextBox text) {
		this(text, false, false);
	}
	
	public TimeValidator(TextBox text, String customMessageKey) {
		this(text, false, false, customMessageKey);
	}
	
	/**
	 * 
	 * @param text The textbox that contains the user input
	 * @param secondsAllowed Should seconds be optional
	 * @param secondsRequired Are seconds required (overwrites the secondsAllowed field)
	 */
	public TimeValidator(TextBox text, boolean secondsAllowed, boolean secondsRequired) {
		this.text = text;
		algorithm = new TimeValidatorAlgorithm(secondsAllowed, secondsRequired);
	}
	
	public TimeValidator(TextBox text, boolean secondsAllowed, boolean secondsRequired, String customMessageKey) {
		this.text = text;
		this.setCustomMsgKey(customMessageKey);
		algorithm = new TimeValidatorAlgorithm(secondsAllowed, secondsRequired);
	}
	
	public TimeValidator(SuggestBox suggest) {
		this(suggest, false, false);
	}
	
	public TimeValidator(SuggestBox suggest, String customMessageKey) {
		this(suggest, false, false, customMessageKey);
		
	}
	
	public TimeValidator(SuggestBox suggest, boolean secondsAllowed, boolean secondsRequired) {
		this.suggest = suggest;
		algorithm = new TimeValidatorAlgorithm(secondsAllowed, secondsRequired);
	}
	
	public TimeValidator(SuggestBox suggest, boolean secondsAllowed, boolean secondsRequired, String customMessageKey) {
		this.suggest = suggest;
		this.setCustomMsgKey(customMessageKey);
		algorithm = new TimeValidatorAlgorithm(secondsAllowed, secondsRequired);
	}
	
	
	@Override
	public void invokeActions(ValidationResult result) {
		if(text != null) {
			for(ValidationAction va : this.getFailureActions())
				va.invoke(result, text);
		}else {
			for(ValidationAction va : this.getFailureActions())
				va.invoke(result, suggest);
		}
			
		
	}

	@Override
	public <V extends ValidationMessages> ValidationResult validate(V messages) {
		String input;
		if(text != null)
			input = text.getText();
		else
			input = suggest.getText();
		
		if(!this.isRequired() && input.equals(""))
			return null;
		
		ValidatorAlgorithmResult result = algorithm.validate(input);
		
		if(result == null)
			return null;
		
		String stdMessage = getStdMessage(messages.getStandardMessages(), result.getErrorCode(), result.getParameters()[0].toString());
		
		return new ValidationResult(this.getErrorMessage(messages, stdMessage, result.getParameters()));
		
	}

	private String getStdMessage(StandardValidationMessages stdMessages,
			int errorCode, String parameter) {
		switch(errorCode) {
		case TimeValidatorAlgorithm.NOT_A_VALID_TIME_WITHOUT_SECONDS:
			return stdMessages.notAValidTimeWithoutSeconds(parameter);
		case TimeValidatorAlgorithm.NOT_A_VALID_TIME_WITH_SECONDS_OPTIONALLY:
			return stdMessages.notAValidTimeWithSecondsOptionally(parameter);
		case TimeValidatorAlgorithm.NOT_A_VALID_TIME_WITH_SECONDS_REQUIRED:
			return stdMessages.notAValidTimeWithSecondsRequired(parameter);
		}
		
		return null;
	}

}
