/*
	Copyright 2009 Anatol Gregory Mayen
	
	Licensed under the Apache License, Version 2.0 (the "License");
	you may not use this file except in compliance with the License. 
	You may obtain a copy of the License at 
	
	http://www.apache.org/licenses/LICENSE-2.0 
	
	Unless required by applicable law or agreed to in writing, software 
	distributed under the License is distributed on an "AS IS" BASIS, 
	WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
	See the License for the specific language governing permissions and 
	limitations under the License. 
*/
package eu.maydu.gwt.validation.client.validators.numeric;

import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.TextBoxBase;

import eu.maydu.gwt.validation.client.ValidationAction;
import eu.maydu.gwt.validation.client.ValidationResult;
import eu.maydu.gwt.validation.client.Validator;
import eu.maydu.gwt.validation.client.i18n.StandardValidationMessages;
import eu.maydu.gwt.validation.client.i18n.ValidationMessages;

/**
 * 
 * @author Anatol Mayen
 *
 */
public class FloatValidator extends Validator<FloatValidator> {

	private TextBoxBase text;
	private SuggestBox suggest;
	private float min, max;
	private boolean noMinMax = false;
	
	public FloatValidator(TextBoxBase text) {
		this(text, null);
	}
	
	public FloatValidator(TextBoxBase text, String customMsgKey) {
		this(text, false);
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public FloatValidator(TextBoxBase text, float min, float max) {
		this(text, min, max, false, null);
	}
	
	public FloatValidator(TextBoxBase text, float min, float max, String customMsgKey) {
		this(text, min, max, false, customMsgKey);
	}
	
	
	public FloatValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain) {
		this(text, preventsPropagationOfValidationChain, null);
	}
	
	public FloatValidator(TextBoxBase text, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new RuntimeException("text must not be null");
		this.text = text;
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public FloatValidator(TextBoxBase text, float min, float max, boolean preventsPropagationOfValidationChain) {
		
		this(text, min, max, preventsPropagationOfValidationChain, null);
		
	}
	
	public FloatValidator(TextBoxBase text, float min, float max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(text == null)
			throw new RuntimeException("text must not be null");
		this.text = text;
		this.min = min;
		this.max = max;
		this.noMinMax = false;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public FloatValidator(SuggestBox suggest) {
		this(suggest, null);
	}
	
	public FloatValidator(SuggestBox suggest, String customMsgKey) {
		this(suggest, false);
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public FloatValidator(SuggestBox suggest, float min, float max) {
		this(suggest, min, max, false);
	}
	
	public FloatValidator(SuggestBox suggest, float min, float max, String customMsgKey) {
		this(suggest, min, max, false, customMsgKey);
	}
	
	
	public FloatValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain) {
		this(suggest, preventsPropagationOfValidationChain, null);
	}
	
	public FloatValidator(SuggestBox suggest, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new RuntimeException("suggest must not be null");
		this.suggest = suggest;
		this.noMinMax = true;
		this.setCustomMsgKey(customMsgKey);
	}
	
	public FloatValidator(SuggestBox suggest, float min, float max, boolean preventsPropagationOfValidationChain) {
		this(suggest, min, max, preventsPropagationOfValidationChain, null);
	}
	
	public FloatValidator(SuggestBox suggest, float min, float max, boolean preventsPropagationOfValidationChain, String customMsgKey) {
		super();
		this.setPreventsPropagationOfValidationChain(preventsPropagationOfValidationChain);
		if(suggest == null)
			throw new RuntimeException("suggest must not be null");
		this.suggest = suggest;
		this.min = min;
		this.max = max;
		this.noMinMax = false;
		this.setCustomMsgKey(customMsgKey);
	}
	
	@Override
	public ValidationResult validate(ValidationMessages allMessages) {
		StandardValidationMessages messages = allMessages.getStandardMessages();
		String str;
		if(text != null)
			str = text.getText();
		else str = suggest.getText();
		if(str == null)
			str = "";
		str = str.trim();
		if(str.equals("") && isRequired())
			return new ValidationResult(getErrorMessage(allMessages, messages.notAnInteger()));
		else if(str.equals(""))
			return null;
		if(!this.noMinMax) {
			//Integer in range
			try {
				double value = Double.parseDouble(str);
				if(value < this.min || value > this.max) {
					return new ValidationResult(getErrorMessage(allMessages, messages.notInRange(this.min, this.max, value),this.min, this.max, value));
				}
			}catch(NumberFormatException ex) {
				return new ValidationResult(getErrorMessage(allMessages, messages.notAFloat()));
			}
		}else {
			//Any integer
			try {
				double value = Double.parseDouble(str);
				if(value > Float.MAX_VALUE || value < Float.MIN_VALUE)
					return new ValidationResult(getErrorMessage(allMessages, messages.notAFloat()));
			}catch(NumberFormatException ex) {
				return new ValidationResult(getErrorMessage(allMessages, messages.notAFloat()));
			}
		}
		return null;
	}
	
	public void invokeActions(ValidationResult result) {
		if(text != null) {
			for(ValidationAction<TextBoxBase> action : getFailureActions())
				action.invoke(result, text);
		}else {
			for(ValidationAction<SuggestBox> action : getFailureActions())
				action.invoke(result, suggest);
		}
	}

}
