/*
 * Copyright (c) 2008-2010, Stephen Colebourne & Michael Nascimento Santos
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  * Neither the name of JSR-310 nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package javax.time.period;

import java.io.Serializable;
import java.util.Collections;
import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.Map.Entry;

import javax.time.CalendricalException;
import javax.time.Duration;
import javax.time.calendar.ISOChronology;
import javax.time.calendar.PeriodUnit;

/**
 * A period of time measured using a number of different units,
 * such as '3 Months, 4 Days and 7 Hours'.
 * <p>
 * {@code PeriodFields} is an immutable period that stores an amount of human-scale
 * time for a number of units. For example, humans typically measure periods of time
 * in units of years, months, days, hours, minutes and seconds. These concepts are
 * defined by instances of {@link PeriodUnit} in the chronology classes. This class
 * allows an amount to be specified for a number of the units, such as '3 Days and 65 Seconds'.
 * <p>
 * Basic mathematical operations are provided - plus(), minus(), multipliedBy(),
 * dividedBy() and negated(), all of which return a new instance
 * <p>
 * A value of zero can also be stored for any unit. This means that a
 * period of zero hours is not equal to a period of zero minutes.
 * However, an empty instance constant exists to represent zero irrespective of unit.
 * The {@link #withZeroesRemoved()} method removes zero values.
 * <p>
 * {@code PeriodFields} can store units of any kind which makes it usable with
 * any calendar system.
 * <p>
 * PeriodFields is immutable and thread-safe.
 *
 * @author Stephen Colebourne
 */
public final class PeriodFields
        implements PeriodProvider, Iterable<PeriodField>, Serializable {

    /**
     * A constant for a period of zero.
     * This constant is independent of any unit.
     */
    public static final PeriodFields ZERO = new PeriodFields(new TreeMap<PeriodUnit, PeriodField>());
    /**
     * The serialization version.
     */
    private static final long serialVersionUID = 1L;

    /**
     * The map of periods.
     */
    private final TreeMap<PeriodUnit, PeriodField> unitFieldMap;

    //-----------------------------------------------------------------------
    /**
     * Obtains a {@code PeriodFields} from an amount and unit.
     * <p>
     * The parameters represent the two parts of a phrase like '6 Days'.
     *
     * @param amount  the amount of create with, positive or negative
     * @param unit  the period unit, not null
     * @return the {@code PeriodFields} instance, never null
     */
    public static PeriodFields of(long amount, PeriodUnit unit) {
        checkNotNull(unit, "PeriodUnit must not be null");
        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
        internalMap.put(unit, PeriodField.of(amount, unit));
        return create(internalMap);
    }

    /**
     * Obtains a {@code PeriodFields} from a single-unit period.
     *
     * @param period  the single-unit period, not null
     * @return the {@code PeriodFields} instance, never null
     */
    public static PeriodFields of(PeriodField period) {
        checkNotNull(period, "PeriodField must not be null");
        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
        internalMap.put(period.getUnit(), period);
        return create(internalMap);
    }

    /**
     * Obtains a {@code PeriodFields} from an array of single-unit periods.
     * <p>
     * The period fields must all have different units.
     *
     * @param periods  the array of single-unit periods, not null
     * @return the {@code PeriodFields} instance, never null
     * @throws IllegalArgumentException if the same period unit occurs twice
     */
    public static PeriodFields of(PeriodField... periods) {
        checkNotNull(periods, "PeriodField array must not be null");
        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
        for (PeriodField period : periods) {
            checkNotNull(period, "PeriodField array must not contain null");
            if (internalMap.put(period.getUnit(), period) != null) {
                throw new IllegalArgumentException("PeriodField array contains the same unit twice");
            }
        }
        return create(internalMap);
    }

//    /**
//     * Obtains a {@code PeriodFields} from an array of single-unit periods.
//     *
//     * @param periods  the array of single-unit periods, not null
//     * @return the {@code PeriodFields} instance, never null
//     * @throws IllegalArgumentException if the same period unit occurs twice
//     */
//    public static PeriodFields of(Iterable<PeriodField> periods) {
//        checkNotNull(periods, "Iterable must not be null");
//        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
//        for (PeriodField period : periods) {
//            checkNotNull(period, "Iterable must not contain null");
//            if (internalMap.put(period.getUnit(), period) != null) {
//                throw new IllegalArgumentException("Iterable contains the same unit twice");
//            }
//        }
//        return create(internalMap);
//    }

    /**
     * Obtains a {@code PeriodFields} from a set of unit-amount pairs.
     * <p>
     * The amount to store for each unit is obtained by calling {@link Number#longValue()}.
     * This will lose any decimal places for instances of {@code Double} and {@code Float}.
     * It may also silently lose precision for instances of {@code BigInteger} or {@code BigDecimal}.
     *
     * @param unitAmountMap  a map of periods that will be used to create this
     *  period, not updated by this method, not null, contains no nulls
     * @return the {@code PeriodFields} instance, never null
     * @throws NullPointerException if the map is null or contains nulls
     */
    public static PeriodFields of(Map<PeriodUnit, ? extends Number> unitAmountMap) {
        checkNotNull(unitAmountMap, "Map must not be null");
        // don't use contains() as tree map and others can throw NPE
        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
        for (Entry<PeriodUnit, ? extends Number> entry : unitAmountMap.entrySet()) {
            PeriodUnit unit = entry.getKey();
            Number amount = entry.getValue();
            checkNotNull(unit, "Null keys are not permitted in unit-amount map");
            checkNotNull(amount, "Null amounts are not permitted in unit-amount map");
            internalMap.put(unit, PeriodField.of(amount.longValue(), unit));
        }
        return create(internalMap);
    }

    //-----------------------------------------------------------------------
    /**
     * Obtains a {@code PeriodFields} from a {@code PeriodProvider}.
     * <p>
     * This factory returns an instance with all the unit-amount pairs from the provider.
     *
     * @param periodProvider  the provider to create from, not null
     * @return the {@code PeriodFields} instance, never null
     * @throws NullPointerException if the period provider is null or returns null
     */
    public static PeriodFields from(PeriodProvider periodProvider) {
        checkNotNull(periodProvider, "PeriodProvider must not be null");
        PeriodFields result = periodProvider.toPeriodFields();
        checkNotNull(result, "PeriodProvider implementation must not return null");
        return result;
    }

    /**
     * Obtains a {@code PeriodFields} by totalling the amounts in a list of
     * {@code PeriodProvider} instances.
     * <p>
     * This method returns a period with all the unit-amount pairs from the providers
     * totalled. Thus a period of '2 Months and 5 Days' combined with a period of
     * '7 Days and 21 Hours' will yield a result of '2 Months, 12 Days and 21 Hours'.
     *
     * @param periodProviders  the providers to total, not null
     * @return the {@code PeriodFields} instance, never null
     * @throws NullPointerException if any period provider is null or returns null
     */
    public static PeriodFields total(PeriodProvider... periodProviders) {
        checkNotNull(periodProviders, "PeriodProvider[] must not be null");
        if (periodProviders.length == 1) {
            return from(periodProviders[0]);
        }
        TreeMap<PeriodUnit, PeriodField> map = createMap();
        for (PeriodProvider periodProvider : periodProviders) {
            PeriodFields periods = from(periodProvider);
            for (PeriodField period : periods.unitFieldMap.values()) {
                PeriodField old = map.get(period.getUnit());
                period = (old != null ? old.plus(period) : period);
                map.put(period.getUnit(), period);
            }
        }
        return create(map);
    }

    //-----------------------------------------------------------------------
    /**
     * Obtains a {@code PeriodFields} from a {@code Duration} based on the standard
     * durations of seconds and nanoseconds.
     * <p>
     * The conversion will create an instance with two units - the {@code ISOChronology}
     * seconds and nanoseconds units. This matches the {@link #toDuration()} method.
     * <p>
     * This conversion can only be used if the duration is being used in a manner
     * compatible with the {@code ISOChronology} definitions of seconds and nanoseconds.
     * This will be the case for most applications - care only needs to be taken if
     * using explicit time-scales.
     *
     * @param duration  the duration to create from, not null
     * @return the {@code PeriodFields} instance, never null
     */
    public static PeriodFields from(Duration duration) {
        checkNotNull(duration, "Duration must not be null");
        TreeMap<PeriodUnit, PeriodField> internalMap = createMap();
        internalMap.put(ISOChronology.periodSeconds(), PeriodField.of(duration.getSeconds(), ISOChronology.periodSeconds()));
        internalMap.put(ISOChronology.periodNanos(), PeriodField.of(duration.getNanosInSecond(), ISOChronology.periodNanos()));
        return create(internalMap);
    }

    //-----------------------------------------------------------------------
    /**
     * Creates a new empty map.
     *
     * @return ordered representation of internal map
     */
    private static TreeMap<PeriodUnit, PeriodField> createMap() {
        return new TreeMap<PeriodUnit, PeriodField>(Collections.reverseOrder());
    }

    /**
     * Internal factory to create an instance using a pre-built map.
     * The map must not be used by the calling code after calling the constructor.
     *
     * @param periodMap  the unit-amount map, not null, assigned not cloned
     * @return the created period, never null
     */
    static PeriodFields create(TreeMap<PeriodUnit, PeriodField> periodMap) {
        if (periodMap.isEmpty()) {
            return ZERO;
        }
        return new PeriodFields(periodMap);
    }

    /**
     * Validates that the input value is not null.
     *
     * @param object  the object to check
     * @param errorMessage  the error to throw
     * @throws NullPointerException if the object is null
     */
    static void checkNotNull(Object object, String errorMessage) {
        if (object == null) {
            throw new NullPointerException(errorMessage);
        }
    }

    //-----------------------------------------------------------------------
    /**
     * Constructs an instance using a pre-built map.
     * The map must not be used by the calling code after calling the constructor.
     *
     * @param periodMap  the map of periods to represent, not null and safe to assign
     */
    private PeriodFields(TreeMap<PeriodUnit, PeriodField> periodMap) {
        this.unitFieldMap = periodMap;
    }

    /**
     * Resolves singletons.
     *
     * @return the resolved instance
     */
    private Object readResolve() {
        if (unitFieldMap.size() == 0) {
            return ZERO;
        }
        return this;
    }

    //-----------------------------------------------------------------------
    /**
     * Checks if this period is zero-length.
     * <p>
     * This checks whether all the amounts in this period are zero.
     *
     * @return true if this period is zero-length
     */
    public boolean isZero() {
        for (PeriodField field : unitFieldMap.values()) {
            if (field.isZero() == false) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks if this period is fully positive, including zero.
     * <p>
     * This checks whether all the amounts in this period are positive,
     * defined as greater than or equal to zero.
     *
     * @return true if this period is fully positive
     */
    public boolean isPositive() {
        for (PeriodField field : unitFieldMap.values()) {
            if (field.isNegative()) {
                return false;
            }
        }
        return true;
    }

    //-----------------------------------------------------------------------
    /**
     * Returns the size of the set of units in this period.
     * <p>
     * This returns the number of different units that are stored.
     *
     * @return number of unit-amount pairs
     */
    public int size() {
        return unitFieldMap.size();
    }

    /**
     * Iterates through all the single-unit periods in this period.
     * <p>
     * This method fulfills the {@link Iterable} interface and allows looping
     * around the contained single-unit periods using the for-each loop.
     *
     * @return an iterator over the single-unit periods in this period, never null
     */
    public Iterator<PeriodField> iterator() {
        return unitFieldMap.values().iterator();
    }

    /**
     * Checks whether this period contains an amount for the unit.
     *
     * @param unit  the unit to query, null returns false
     * @return true if the map contains an amount for the unit
     */
    public boolean contains(PeriodUnit unit) {
        return unitFieldMap.containsKey(unit);
    }

    //-----------------------------------------------------------------------
    /**
     * Gets the period for the specified unit.
     * <p>
     * This method allows the period to be queried by unit, like a map.
     * If the unit is not found then {@code null} is returned.
     *
     * @param unit  the unit to query, not null
     * @return the period, null if no period stored for the unit
     */
    public PeriodField get(PeriodUnit unit) {
        checkNotNull(unit, "PeriodUnit must not be null");
        return unitFieldMap.get(unit);
    }

    /**
     * Gets the amount of this period for the specified unit.
     * <p>
     * This method allows the amount to be queried by unit, like a map.
     * If the unit is not found then zero is returned.
     *
     * @param unit  the unit to query, not null
     * @return the period amount, 0 if no period stored for the unit
     * @throws CalendricalException if there is no amount for the unit
     */
    public long getAmount(PeriodUnit unit) {
        PeriodField field = get(unit);
        if (field == null) {
            return 0;
        }
        return field.getAmount();
    }

    /**
     * Gets the amount of this period for the specified unit converted
     * to an {@code int}.
     * <p>
     * This method allows the amount to be queried by unit, like a map.
     * If the unit is not found then zero is returned.
     *
     * @param unit  the unit to query, not null
     * @return the period amount, 0 if no period stored for the unit
     * @throws CalendricalException if there is no amount for the unit
     * @throws ArithmeticException if the amount is too large to be returned in an int
     */
    public int getAmountInt(PeriodUnit unit) {
        PeriodField field = get(unit);
        if (field == null) {
            return 0;
        }
        return field.getAmountInt();
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a copy of this period with all zero amounts removed.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @return a {@code PeriodField} based on this period with zero amounts removed, never null
     */
    public PeriodFields withZeroesRemoved() {
        if (isZero()) {
            return ZERO;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        for (Iterator<PeriodField> it = copy.values().iterator(); it.hasNext(); ) {
            if (it.next().isZero()) {
                it.remove();
            }
        }
        return create(copy);
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a copy of this period with the specified amount for the unit.
     * <p>
     * If this period already contains an amount for the unit then the amount
     * is replaced. Otherwise, the unit-amount pair is added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param amount  the amount to store in terms of the unit, positive or negative
     * @param unit  the unit to store not null
     * @return a {@code PeriodField} based on this period with the specified period overlaid, never null
     */
    public PeriodFields with(long amount, PeriodUnit unit) {
        PeriodField existing = get(unit);
        if (existing != null && existing.getAmount() == amount) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        copy.put(unit, PeriodField.of(amount, unit));
        return create(copy);
    }

    /**
     * Returns a copy of this period with the specified values altered.
     * <p>
     * This method operates on each unit in the input in turn.
     * If this period already contains an amount for the unit then the amount
     * is replaced. Otherwise, the unit-amount pair is added.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param period  the period to store, not null
     * @return a {@code PeriodField} based on this period with the specified period overlaid, never null
     */
    public PeriodFields with(PeriodFields period) {
        if (this == ZERO) {
            return period;
        }
        if (period == ZERO) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        copy.putAll(period.unitFieldMap);
        return create(copy);
    }

    /**
     * Returns a copy of this period with the specified unit removed.
     * <p>
     * If this period already contains an amount for the unit then the amount
     * is removed. Otherwise, no action occurs.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param unit  the unit to remove, not null
     * @return a {@code PeriodField} based on this period with the specified unit removed, never null
     */
    public PeriodFields without(PeriodUnit unit) {
        checkNotNull(unit, "PeriodUnit must not be null");
        if (unitFieldMap.containsKey(unit) == false) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        copy.remove(unit);
        return create(copy);
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a copy of this period with the specified period added.
     * <p>
     * The returned period will take each unit in the provider and add the value
     * to the amount already stored in this period, returning a new one.
     * If this period does not contain an amount for the unit then the unit and
     * amount are simply returned directly in the result. The result will have
     * the union of the units in this instance and the units in the specified instance.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param periodProvider  the period to add, not null
     * @return a {@code PeriodField} based on this period with the specified period added, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields plus(PeriodProvider periodProvider) {
        checkNotNull(periodProvider, "PeriodProvider must not be null");
        if (this == ZERO && periodProvider instanceof PeriodFields) {
            return (PeriodFields) periodProvider;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        PeriodFields periods = from(periodProvider);
        for (PeriodField period : periods.unitFieldMap.values()) {
            PeriodField old = copy.get(period.getUnit());
            period = (old != null ? old.plus(period) : period);
            copy.put(period.getUnit(), period);
        }
        return create(copy);
    }

    /**
     * Returns a copy of this period with the specified period added.
     * <p>
     * The result will contain the units and amounts from this period plus the
     * specified unit and amount.
     * The specified unit will always be in the result even if the amount is zero.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param amount  the amount to add, measured in the specified unit, positive or negative
     * @param unit  the unit defining the amount, not null
     * @return a {@code PeriodField} based on this period with the specified period added, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields plus(long amount, PeriodUnit unit) {
        checkNotNull(unit, "PeiodRule must not be null");
        if (amount == 0 && contains(unit)) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        PeriodField old = copy.get(unit);
        PeriodField field = (old != null ? old.plus(amount) : PeriodField.of(amount, unit));
        copy.put(unit, field);
        return create(copy);
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a copy of this period with the specified period subtracted.
     * <p>
     * The returned period will take each unit in the provider and subtract the
     * value from the amount already stored in this period, returning a new one.
     * If this period does not contain an amount for the unit then the unit and
     * amount are simply returned directly in the result. The result will have
     * the union of the units in this instance and the units in the specified instance.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param period  the period to subtract, not null
     * @return a {@code PeriodField} based on this period with the specified period subtracted, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields minus(PeriodProvider periodProvider) {
        checkNotNull(periodProvider, "PeriodProvider must not be null");
        if (this == ZERO && periodProvider instanceof PeriodFields) {
            return (PeriodFields) periodProvider;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        PeriodFields periods = from(periodProvider);
        for (PeriodField period : periods.unitFieldMap.values()) {
            PeriodField old = copy.get(period.getUnit());
            period = (old != null ? old.minus(period) : period.negated());
            copy.put(period.getUnit(), period);
        }
        return create(copy);
    }

    /**
     * Returns a copy of this period with the specified period subtracted.
     * <p>
     * The result will contain the units and amounts from this period minus the
     * specified unit and amount.
     * The specified unit will always be in the result even if the amount is zero.
     * <p>
     * This instance is immutable and unaffected by this method call.
     *
     * @param amount  the amount to subtract, measured in the specified unit, positive or negative
     * @param unit  the unit defining the amount, not null
     * @return a {@code PeriodField} based on this period with the specified period subtracted, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields minus(long amount, PeriodUnit unit) {
        checkNotNull(unit, "PeiodRule must not be null");
        if (amount == 0 && contains(unit)) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = clonedMap();
        PeriodField old = copy.get(unit);
        copy.put(unit, old != null ? old.minus(amount) : PeriodField.of(amount, unit).negated());
        return create(copy);
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a new instance with each amount in this period multiplied
     * by the specified scalar.
     *
     * @param scalar  the scalar to multiply by, not null
     * @return a {@code PeriodField} based on this period with the amount multiplied by the scalar, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields multipliedBy(long scalar) {
        if (scalar == 1 || isZero()) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = createMap();
        for (PeriodField field : this) {
            copy.put(field.getUnit(), field.multipliedBy(scalar));
        }
        return create(copy);
    }

    /**
     * Returns a new instance with each amount in this period divided
     * by the specified value.
     *
     * @param divisor  the value to divide by, not null, not zero
     * @return a {@code PeriodField} based on this period with the amount divided by the divisor, never null
     * @throws ArithmeticException if dividing by zero
     */
    public PeriodFields dividedBy(long divisor) {
        if (divisor == 0) {
            throw new ArithmeticException("Cannot divide by zero");
        }
        if (divisor == 1 || isZero()) {
            return this;
        }
        TreeMap<PeriodUnit, PeriodField> copy = createMap();
        for (PeriodField field : this) {
            copy.put(field.getUnit(), field.dividedBy(divisor));
        }
        return create(copy);
    }

    /**
     * Returns a new instance with each amount in this period negated.
     *
     * @return a {@code PeriodField} based on this period with the amount negated, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields negated() {
        return multipliedBy(-1);
    }

    //-----------------------------------------------------------------------
    /**
     * Clone the internal data storage map.
     *
     * @return the cloned map, never null
     */
    @SuppressWarnings("unchecked")
    private TreeMap<PeriodUnit, PeriodField> clonedMap() {
        return (TreeMap) unitFieldMap.clone();
    }

    //-----------------------------------------------------------------------
    /**
     * Converts this period to one containing only the units specified.
     * <p>
     * This will attempt to convert this period to each of the specified units
     * in turn. It is recommended to specify the units from largest to smallest.
     * If this period is already one of the specified units, then {@code this}
     * is returned.
     * <p>
     * For example, '3 Hours' can normally be converted to both minutes and seconds.
     * If the units array contains both 'Minutes' and 'Seconds', then the result will
     * be measured in whichever is first in the array.
     * <p>
     * A total of a compound period can also be obtained.
     * For example, '3 Hours, 34 Minutes' can be totalled to minutes by passing the
     * single unit of minutes, resulting in '214 Minutes'.
     *
     * @param units  the required unit array, not altered, not null
     * @return a {@code PeriodField} equivalent to this period, never null
     * @throws CalendricalException if the period cannot be converted to any of the units
     * @throws ArithmeticException if the calculation overflows
     */
    public PeriodFields toEquivalentPeriod(PeriodUnit... units) {
        TreeMap<PeriodUnit, PeriodField> map = createMap();
        for (PeriodField period : unitFieldMap.values()) {
            period = period.toEquivalentPeriod(units);
            PeriodField old = map.get(period.getUnit());
            period = (old != null ? old.plus(period) : period);
            map.put(period.getUnit(), period);
        }
        return (map.equals(unitFieldMap) ? this : create(map));
    }

    //-----------------------------------------------------------------------
    /**
     * Converts this object to a map of units to amounts.
     * <p>
     * The returned map will never be null, however it may be empty.
     * It is sorted by the unit, returning the largest first.
     * It is independent of this object - changes will not be reflected back.
     *
     * @return the independent, modifiable map of periods, never null, never contains null
     */
    public SortedMap<PeriodUnit, Long> toRuleAmountMap() {
        SortedMap<PeriodUnit, Long> map = new TreeMap<PeriodUnit, Long>(Collections.reverseOrder());
        for (PeriodField field : this) {
            map.put(field.getUnit(), field.getAmount());
        }
        return map;
    }

    /**
     * Converts this period to an estimated duration.
     * <p>
     * Each {@link PeriodUnit} contains an estimated duration for that unit.
     * This method uses that estimate to calculate a total estimated duration for
     * this period.
     *
     * @return the estimated duration of this period, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public Duration toEstimatedDuration() {
        Duration dur = Duration.ZERO;
        for (PeriodField field : this) {
            dur = dur.plus(field.toEstimatedDuration());
        }
        return dur;
    }

    /**
     * Converts this {@code PeriodFields} to a {@code Duration} based on the standard
     * durations of seconds and nanoseconds.
     * <p>
     * The conversion is based on the {@code ISOChronology} definition of the seconds and
     * nanoseconds units. If all the fields in this period can be converted to one of these
     * units then the conversion will succeed, subject to calculation overflow.
     * If any field cannot be converted to these fields above then an exception is thrown.
     * <p>
     * This conversion can only be used if the duration is being used in a manner
     * compatible with the {@code ISOChronology} definitions of seconds and nanoseconds.
     * This will be the case for most applications - care only needs to be taken if
     * using explicit time-scales.
     *
     * @return the duration of this period based on {@code ISOChronology} fields, never null
     * @throws ArithmeticException if the calculation overflows
     */
    public Duration toDuration() {
        PeriodFields period = toEquivalentPeriod(ISOChronology.periodSeconds(), ISOChronology.periodNanos());
        return Duration.seconds(period.getAmount(ISOChronology.periodSeconds()), period.getAmount(ISOChronology.periodNanos()));
    }

    //-----------------------------------------------------------------------
    /**
     * Converts this period to a {@code PeriodFields}, trivially
     * returning {@code this}.
     *
     * @return {@code this}, never null
     */
    public PeriodFields toPeriodFields() {
        return this;
    }

    //-----------------------------------------------------------------------
    /**
     * Checks if this instance equal to the object specified.
     * <p>
     * Two {@code PeriodFields} instances are equal if all the contained
     * {@code PeriodField} instances are equal.
     *
     * @param obj  the other period to compare to, null returns false
     * @return true if this instance is equal to the specified period
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof PeriodFields) {
            PeriodFields other = (PeriodFields) obj;
            return unitFieldMap.equals(other.unitFieldMap);
        }
        return false;
    }

    /**
     * Returns the hash code for this period.
     *
     * @return a suitable hash code
     */
    @Override
    public int hashCode() {
        return unitFieldMap.hashCode();
    }

    //-----------------------------------------------------------------------
    /**
     * Returns a string representation of the period, such as '[6 Days, 13 Hours]'.
     *
     * @return a descriptive representation of the period, not null
     */
    @Override
    public String toString() {
        if (unitFieldMap.size() == 0) {
            return "[]";
        }
        StringBuilder buf = new StringBuilder();
        buf.append('[');
        for (PeriodField field : this) {
            buf.append(field.toString()).append(',').append(' '); 
        }
        buf.setLength(buf.length() - 2);
        buf.append(']');
        return buf.toString();
    }

}
