/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.consent.storage.impl;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import jakarta.json.JsonArray;
import jakarta.json.JsonException;
import jakarta.json.JsonReader;
import jakarta.json.JsonReaderFactory;
import jakarta.json.JsonString;
import jakarta.json.JsonStructure;
import jakarta.json.JsonValue;
import jakarta.json.JsonValue.ValueType;
import jakarta.json.spi.JsonProvider;
import jakarta.json.stream.JsonGenerator;
import jakarta.json.stream.JsonGeneratorFactory;

import org.opensaml.storage.StorageSerializer;
import org.slf4j.Logger;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.component.AbstractInitializableComponent;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Serializes a {@link Collection} of strings. <code>Null</code> elements and non-string values are ignored.
 */
public class CollectionSerializer extends AbstractInitializableComponent implements
        StorageSerializer<Collection<String>> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(CollectionSerializer.class);

    /** JSON generator factory. */
    @Nonnull private final JsonGeneratorFactory generatorFactory;

    /** JSON reader factory. */
    @Nonnull private final JsonReaderFactory readerFactory;

    /** Constructor. */
    @SuppressWarnings("null")
    public CollectionSerializer() {
        final JsonProvider provider = JsonProvider.provider();
        generatorFactory = provider.createGeneratorFactory(null);
        readerFactory = provider.createReaderFactory(null);
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull @NotEmpty public String serialize(@Nonnull final Collection<String> instance) throws IOException {
        Constraint.isNotNull(instance, "Storage indexes cannot be null");

        final StringWriter sink = new StringWriter(128);
        
        try (final JsonGenerator gen = generatorFactory.createGenerator(sink)) {
            gen.writeStartArray();
            for (final String element : instance) {
                if (element != null) {
                    gen.write(element);
                }
            }
            gen.writeEnd();
        }

        final String serialized = sink.toString();
        log.debug("Serialized '{}' as '{}'", instance, serialized);
        return serialized;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull public Collection<String> deserialize(final long version,
            @Nonnull @NotEmpty final String context, @Nonnull @NotEmpty final String key,
            @Nonnull @NotEmpty final String value, @Nullable final Long expiration) throws IOException {

        try (final JsonReader reader = readerFactory.createReader(new StringReader(value))) {
            final JsonStructure st = reader.read();
            if (!(st instanceof JsonArray)) {
                throw new IOException("Found invalid data structure");
            }

            final Collection<String> collection = new ArrayList<>();

            for (final JsonValue arrayValue : (JsonArray) st) {
                if (arrayValue.getValueType().equals(ValueType.STRING)) {
                    collection.add(((JsonString) arrayValue).getString());
                }
            }

            log.debug("Deserialized context '{}' key '{}' value '{}' expiration '{}' as '{}'",
                    new Object[] { context, key, value, expiration, collection, });
            return collection;
        } catch (final NullPointerException | ClassCastException | ArithmeticException | JsonException e) {
            log.error("Exception while parsing consent: {}", e.getMessage());
            throw new IOException("Found invalid data structure while parsing consent", e);
        }
    }

}