/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.saml1.profile.config.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.saml.profile.config.impl.AbstractSAMLProfileConfiguration;
import net.shibboleth.saml.profile.config.SAMLArtifactAwareProfileConfiguration;
import net.shibboleth.saml.profile.config.SAMLArtifactConfiguration;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.FunctionSupport;

/**
 * Configuration support for artifact-aware profiles.
 * 
 * @since 3.4.0
 */
public abstract class AbstractSAML1ArtifactAwareProfileConfiguration extends AbstractSAMLProfileConfiguration
        implements SAMLArtifactAwareProfileConfiguration {
    
    /** Lookup function to supply artifactConfiguration property. */
    @Nonnull private Function<ProfileRequestContext,SAMLArtifactConfiguration> artifactConfigurationLookupStrategy;

    /**
     * Constructor.
     * 
     * @param profileId unique ID for this profile
     */
    protected AbstractSAML1ArtifactAwareProfileConfiguration(@Nonnull @NotEmpty final String profileId) {
        super(profileId);
        
        artifactConfigurationLookupStrategy = FunctionSupport.constant(null);
    }
    
    /** {@inheritDoc} */
    @Nullable public SAMLArtifactConfiguration getArtifactConfiguration(
            @Nullable final ProfileRequestContext profileRequestContext) {
        return artifactConfigurationLookupStrategy.apply(profileRequestContext);
    }

    /**
     * Set the SAML artifact configuration, if any.
     * 
     * @param config configuration to set
     */
    public void setArtifactConfiguration(@Nullable final SAMLArtifactConfiguration config) {
        artifactConfigurationLookupStrategy = FunctionSupport.constant(config);
    }

    /**
     * Set a lookup strategy for the SAML artifact configuration.
     *
     * @param strategy  lookup strategy
     * 
     * @since 3.3.0
     */
    public void setArtifactConfigurationLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,SAMLArtifactConfiguration> strategy) {
        artifactConfigurationLookupStrategy = Constraint.isNotNull(strategy, "Lookup strategy cannot be null");
    }

}