/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.saml1.profile.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.Nonnull;

import org.opensaml.messaging.context.navigate.ChildContextLookup;
import org.opensaml.messaging.context.navigate.MessageLookup;
import org.opensaml.profile.action.ActionSupport;
import org.opensaml.profile.action.EventIds;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.profile.context.navigate.InboundMessageContextLookup;
import org.opensaml.saml.saml1.core.AttributeDesignator;
import org.opensaml.saml.saml1.core.AttributeQuery;
import org.opensaml.saml.saml1.core.Request;
import org.slf4j.Logger;

import net.shibboleth.idp.attribute.AttributeDecodingException;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.context.AttributeContext;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoder;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscoderSupport;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.idp.profile.AbstractProfileAction;
import net.shibboleth.profile.context.RelyingPartyContext;
import net.shibboleth.shared.annotation.constraint.Live;
import net.shibboleth.shared.annotation.constraint.NonnullAfterInit;
import net.shibboleth.shared.annotation.constraint.NonnullBeforeExec;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.service.ReloadableService;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.service.ServiceableComponent;

/**
 * Action that filters a set of attributes against the {@link org.opensaml.saml.saml1.core.AttributeDesignator}
 * objects in an {@link AttributeQuery}.
 * 
 * @event {@link org.opensaml.profile.action.EventIds#PROCEED_EVENT_ID}
 */
public class FilterByQueriedAttributeDesignators extends AbstractProfileAction {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(FilterByQueriedAttributeDesignators.class);

    /** Transcoder registry service object. */
    @NonnullAfterInit private ReloadableService<AttributeTranscoderRegistry> transcoderRegistry;
    
    /** Strategy used to locate the {@link Request} containing the query to filter against. */
    @Nonnull private Function<ProfileRequestContext,Request> requestLookupStrategy;

    /** Strategy used to locate the {@link AttributeContext} to filter. */
    @Nonnull private Function<ProfileRequestContext,AttributeContext> attributeContextLookupStrategy;

    /** Query to filter against. */
    @NonnullBeforeExec private AttributeQuery query;
    
    /** AttributeContext to filter. */
    @NonnullBeforeExec private AttributeContext attributeContext;

    /** Constructor. */
    public FilterByQueriedAttributeDesignators() {
        final Function<ProfileRequestContext,AttributeContext> acls =
                new ChildContextLookup<>(AttributeContext.class).compose(
                        new ChildContextLookup<>(RelyingPartyContext.class));
        final Function<ProfileRequestContext,Request> rls =
                new MessageLookup<>(Request.class).compose(new InboundMessageContextLookup()); 
        assert acls != null && rls != null;
        attributeContextLookupStrategy = acls;
        requestLookupStrategy = rls;
    }

    /**
     * Sets the registry of transcoding rules to apply to encode attributes.
     * 
     * @param registry registry service interface
     */
    public void setTranscoderRegistry(@Nonnull final ReloadableService<AttributeTranscoderRegistry> registry) {
        checkSetterPreconditions();
        transcoderRegistry = Constraint.isNotNull(registry, "AttributeTranscoderRegistry cannot be null");
    }
    
    /**
     * Set the strategy used to locate the {@link Request} associated with a given {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setRequestLookupStrategy(@Nonnull final Function<ProfileRequestContext,Request> strategy) {
        checkSetterPreconditions();
        requestLookupStrategy = Constraint.isNotNull(strategy, "Request lookup strategy cannot be null");
    }

    /**
     * Set the strategy used to locate the {@link AttributeContext} associated with a given
     * {@link ProfileRequestContext}.
     * 
     * @param strategy lookup strategy
     */
    public void setAttributeContextLookupStrategy(
            @Nonnull final Function<ProfileRequestContext,AttributeContext> strategy) {
        checkSetterPreconditions();
        attributeContextLookupStrategy =
                Constraint.isNotNull(strategy, "AttributeContext lookup strategy cannot be null");
    }

    /** {@inheritDoc} */
    @Override
    protected void doInitialize() throws ComponentInitializationException {
        super.doInitialize();
        
        if (transcoderRegistry == null) {
            throw new ComponentInitializationException("AttributeTranscoderRegistry cannot be null");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    protected boolean doPreExecute(@Nonnull final ProfileRequestContext profileRequestContext) {
        
        if (!super.doPreExecute(profileRequestContext)) {
            return false;
        }
        
        final Request request = requestLookupStrategy.apply(profileRequestContext);
        if (request != null) {
            query = request.getAttributeQuery();
        }
        final AttributeQuery ac = query;
        
        final List<AttributeDesignator> designators = ac == null ? null : ac.getAttributeDesignators();
        if (ac == null || designators == null|| designators.isEmpty()) {
            log.debug("No AttributeDesignators found, nothing to do ");
            return false;
        }
        
        attributeContext = attributeContextLookupStrategy.apply(profileRequestContext);
        if (attributeContext == null) {
            log.debug("{} No attribute context, no attributes to filter", getLogPrefix());
            return false;
        }
        assert attributeContext!=null;
        if (attributeContext.getIdPAttributes().isEmpty()) {
            log.debug("{} No attributes to filter", getLogPrefix());
            return false;
        }

        return true;
    }
    
    /** {@inheritDoc} */
    @Override
    protected void doExecute(@Nonnull final ProfileRequestContext profileRequestContext) {

        final Set<String> decodedAttributeIds = new HashSet<>();

        try (final ServiceableComponent<AttributeTranscoderRegistry> component =
                transcoderRegistry.getServiceableComponent()) {
            for (final AttributeDesignator designator : query.getAttributeDesignators()) {
                assert designator!=null;
                try {
                    decodeAttributeDesignator(component.getComponent(), profileRequestContext, designator,
                            decodedAttributeIds);
                } catch (final AttributeDecodingException e) {
                    log.warn("{} Error decoding AttributeDesignators", getLogPrefix(), e);
                }
            }
        } catch (final ServiceException e) {
            log.error("Attribute transcoder service unavailable", e);
            ActionSupport.buildEvent(profileRequestContext, EventIds.MESSAGE_PROC_ERROR);
            return;
        }

        final Collection<IdPAttribute> keepers = new ArrayList<>(query.getAttributeDesignators().size());
        log.debug("Query content mapped to attribute IDs: {}", decodedAttributeIds);
        
        for (final IdPAttribute attribute : attributeContext.getIdPAttributes().values()) {
            if (decodedAttributeIds.contains(attribute.getId())) {
                log.debug("Retaining attribute '{}' requested by query", attribute.getId());
                keepers.add(attribute);
            } else {
                log.debug("Removing attribute '{}' not requested by query", attribute.getId());
            }
        }
        
        attributeContext.setIdPAttributes(keepers);
    }

    /**
     * Access the registry of transcoding rules to decode the input {@link AttributeDesignator}.
     * 
     * @param registry  registry of transcoding rules
     * @param profileRequestContext current profile request context
     * @param input input object
     * @param results collection to add attributeIDs to
     * 
     * @throws AttributeDecodingException if an error occurs or no results were obtained
     */
    private void decodeAttributeDesignator(@Nonnull final AttributeTranscoderRegistry registry,
            @Nonnull final ProfileRequestContext profileRequestContext, @Nonnull final AttributeDesignator input,
            @Nonnull @Live final Collection<String> results) throws AttributeDecodingException {
        
        final Collection<TranscodingRule> transcodingRules = registry.getTranscodingRules(input);
        if (transcodingRules.isEmpty()) {
            throw new AttributeDecodingException("No transcoding rule for AttributeDesignator '" +
                    input.getAttributeName() + "'");
        }
        
        for (final TranscodingRule rules : transcodingRules) {
            assert rules != null;
            final AttributeTranscoder<AttributeDesignator> transcoder = TranscoderSupport.getTranscoder(rules);
            final IdPAttribute decodedAttribute = transcoder.decode(profileRequestContext, input, rules);
            if (decodedAttribute != null) {
                results.add(decodedAttribute.getId());
            }
        }
    }

}