/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.transcoding.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry.NamingFunction;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.NonnullFunction;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.spring.service.AbstractServiceableComponent;
import net.shibboleth.shared.spring.service.impl.SpringServiceableComponent;

/**
 * Strategy for summoning up an {@link AttributeTranscoderRegistryImpl} from a populated {@link ApplicationContext}.
 */
public class AttributeRegistryServiceStrategy extends AbstractIdentifiableInitializableComponent implements
        NonnullFunction<ApplicationContext,AbstractServiceableComponent<AttributeTranscoderRegistry>> {

    /** Name of bean to supply naming function registry property. */
    @Nullable private Collection<NamingFunction<?>> namingRegistry;
    
    /** Optional factory function for building extended activation conditions. */
    @Nullable private Function<Map<String,Object>,Predicate<ProfileRequestContext>> extendedConditionFactory;
    
    /**
     * Sets the collection of {@link NamingFunction}s to install into the registry.
     * 
     * <p>This is done for auto-wiring exposure since plugins may be supplying additional functions
     * outside the registry's own service context.</p>
     * 
     * @param namingFunctions collection of functions to install
     */
    @Autowired
    public void setNamingRegistry(@Nullable final Collection<NamingFunction<?>> namingFunctions) {
        checkSetterPreconditions();
        
        if (namingFunctions != null) {
            namingRegistry = CollectionSupport.copyToList(namingFunctions);
        } else {
            namingRegistry = CollectionSupport.emptyList();
        }
    }

    /**
     * Set a factory function to call to consume a rule set and produce an optional "extended"
     * activation condition {@link Predicate}.
     * 
     * <p>This allows deployments to define extended property names that represent condition types not
     * known to this layer of the code base.</p>
     * 
     * @param factory factory function
     */
    public void setExtendedConditionFactory(
            @Nullable final Function<Map<String,Object>,Predicate<ProfileRequestContext>> factory) {
        checkSetterPreconditions();
        
        extendedConditionFactory = factory;
    }

    /** {@inheritDoc} */
    @Nonnull public AbstractServiceableComponent<AttributeTranscoderRegistry> apply(
            @Nullable final ApplicationContext appContext) {
        checkComponentActive();
        
        if (appContext == null) {
            throw new ServiceException("ApplicationContext was null");
        }
        
        final Collection<TranscodingRule> mappingBeans = appContext.getBeansOfType(TranscodingRule.class).values();
        final Collection<TranscodingRuleLoader> loaderBeans =
                appContext.getBeansOfType(TranscodingRuleLoader.class).values();

        final Collection<TranscodingRule> holder = new ArrayList<>();
        if (mappingBeans != null) {
            holder.addAll(mappingBeans);
        }
        if (loaderBeans != null) {
            loaderBeans.forEach(loader -> holder.addAll(loader.getRules()));
        }
        
        final AttributeTranscoderRegistryImpl registry = new AttributeTranscoderRegistryImpl();
        registry.setId(ensureId());
        registry.setApplicationContext(appContext);
        registry.setNamingRegistry(namingRegistry);
        registry.setExtendedConditionFactory(extendedConditionFactory);
        registry.setTranscoderRegistry(holder);
        
        final SpringServiceableComponent<AttributeTranscoderRegistry> result;

        try {
            registry.initialize();
            result = new SpringServiceableComponent<>(registry);
            result.setApplicationContext(appContext);
            result.initialize();
            return result;
        } catch (final ComponentInitializationException e) {
            throw new ServiceException("Unable to initialize attribute transcoder registry for "
                    + appContext.getDisplayName(), e);
        }
    }
    
}