/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.transcoding.impl;

import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.core.io.FileSystemResource;

import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.logic.PredicateSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.spring.resource.ResourceHelper;

/**
 * A mechanism for loading a set of {@link TranscodingRule} objects from sources such as maps
 * or directories of property files.
 */
public class TranscodingRuleLoader {
    
    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(TranscodingRuleLoader.class);
    
    /** Rules loaded. */
    @Nonnull private final Collection<TranscodingRule> rules;
    
    /**
     * Load rules from all files found below a directory root.
     * 
     * <p>Files are assumed to be Java property files in text format.</p>
     * 
     * <p>Individual rules that fail to load will be skipped.</p>
     * 
     * <p>The file extensions must include the period, and only apply to
     * files, while all directories will be examined.</p>
     * 
     * @param dir root to search
     * @param extensions file extensions to include
     * 
     * @throws IOException if an error occurs
     */
    public TranscodingRuleLoader(@Nonnull @ParameterName(name="dir") final Path dir,
            @Nullable @ParameterName(name="extensions") final Collection<String> extensions)
                    throws IOException {
        log.debug("Loading rules from directory ({})", dir);
        final Collection<TranscodingRule> holder = new ArrayList<>();
        
        try (final DirectoryStream<Path> dirstream  = Files.newDirectoryStream(dir)) {
            for (final Path child : dirstream) {
                final File file =  child.toFile();
                if (file.isDirectory()) {
                    try {
                        holder.addAll(new TranscodingRuleLoader(child, extensions).getRules());
                    } catch (final IOException e) {
                        log.error("Failed to load rules from directory ({})", file, e);
                    }
                } else if (extensions == null || extensions.isEmpty() ||
                        PredicateSupport.anyMatch((String ext) -> file.getName().endsWith(ext)).test(extensions)) {
                    log.debug("Loading rule from property set in file ({})", file);
                    try {
                        final TranscodingRule rule =
                                TranscodingRule.fromResource(ResourceHelper.of(new FileSystemResource(file)));
                        if (rule.getMap().isEmpty()) {
                            log.info("Transcoding file {} contained no rules", child);
                        } else {
                            holder.add(rule);
                        }
                    } catch (final IOException e) {
                        log.error("Failed to load rule from file ({})", file, e);
                    }
                } else {
                    log.debug("Ignoring file ({}) with non-matching extension", file);
                }
            }
        }
        
        rules = CollectionSupport.copyToList(holder);
    }

    /**
     * Load rules from all files found below a directory root.
     * 
     * <p>Files are assumed to be Java property files in text format.</p>
     * 
     * <p>Individual rules that fail to load will be skipped.</p>
     * 
     * @param dir root to search
     * 
     * @throws IOException if an error occurs
     */
    public TranscodingRuleLoader(@Nonnull @ParameterName(name="dir") final Path dir) throws IOException {
        this(dir, null);
    }
    
    /**
     * Constructor.
     *
     * @param maps a collection of maps to build rules around directly.
     */
    public TranscodingRuleLoader(@Nonnull @ParameterName(name="maps") final Collection<Map<String,Object>> maps) {
        Constraint.isNotNull(maps, "Input collection cannot be null");
        
        rules = maps
                .stream()
                .map(m -> {
                    assert m!= null;
                    return new TranscodingRule(m);
                    })
                .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableList()))
                .get();
    }

    /**
     * Get the rules loaded by this object.
     * 
     * @return collection of rules
     */
    @Nonnull @Unmodifiable @NotLive public Collection<TranscodingRule> getRules() {
        return rules;
    }
    
}