/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring;

import java.util.List;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.logic.ResolutionLabelPredicate;
import net.shibboleth.idp.attribute.resolver.spring.impl.InputAttributeDefinitionParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.InputDataConnectorParser;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.PredicateSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.custom.AbstractCustomBeanDefinitionParser;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ElementSupport;

/** Bean definition parser for a {@link net.shibboleth.idp.attribute.resolver.ResolverPlugin}. */
public abstract class BaseResolverPluginParser extends AbstractCustomBeanDefinitionParser {

    /** Stores class of the predicate handling the relyingParties setting. */
    @Nullable private Class<? extends Predicate<?>> relyingPartyPredicateClass;
    
    /** Stores name of factory method to invoke on the predicate class handling the relyingParties setting. */
    @Nullable private String relyingPartyPredicateFactoryMethod;
    
    /** An Id for the definition, used for debugging messages and creating names of children. */
    @Nonnull @NotEmpty private String defnId = "<Unnamed Attribute or Connector>";

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(BaseResolverPluginParser.class);

    /** Constructor. */
    @SuppressWarnings("unchecked")
    public BaseResolverPluginParser() {
        try {
            final String className = getCustomProperty(BaseResolverPluginParser.class.getName()
                    + ".RelyingPartyIdPredicate.class", null);
            if (className != null) {
                relyingPartyPredicateClass = (Class<? extends Predicate<?>>) Class.forName(className);
                log.trace("Using class for mapped tag predicate: {}", className);
                relyingPartyPredicateFactoryMethod = getCustomProperty(BaseResolverPluginParser.class.getName()
                        + ".RelyingPartyIdPredicate.factoryMethod", null);
            }
        } catch (final ClassNotFoundException e) {
            // Will warn later any time we encounter a relevant setting.
        }
    }
    
    /**
     * Helper for logging.
     * 
     * @return the definition ID
     */
    @Nonnull @NotEmpty protected String getDefinitionId() {
        return defnId;
    }

// Checkstyle: CyclomaticComplexity|MethodLength OFF
    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(config, parserContext, builder);
        final String id = StringSupport.trimOrNull(config.getAttributeNS(null, "id"));
        log.debug("Parsing configuration for {} plugin with id: {}", config.getLocalName(), id);
        builder.addPropertyValue("id", id);
        if (null != id) {
            defnId = id;
        }
        builder.setInitMethodName("initialize");
        builder.setDestroyMethodName("destroy");

        if (config.hasAttributeNS(null, "activationConditionRef")) {
            if (config.hasAttributeNS(null, "relyingParties") ||
                    config.hasAttributeNS(null, "resolutionPhases") ||
                    config.hasAttributeNS(null, "excludeRelyingParties") ||
                    config.hasAttributeNS(null, "excludeResolutionPhases")) {
                log.warn("relyingParties/resolutionPhases and variants ignored, using activationConditionRef");
            }
            builder.addPropertyReference("activationCondition",
                    AttributeSupport.ensureAttributeValue(config, null, "activationConditionRef"));
        } else {
            final BeanDefinitionBuilder condition = getActivationCondition(config);
            if (condition != null) {
                builder.addPropertyValue("activationCondition", condition.getBeanDefinition());
            }
        }

        if (config.hasAttributeNS(null, "propagateResolutionExceptions")) {
            builder.addPropertyValue("propagateResolutionExceptions",
                    StringSupport.trimOrNull(config.getAttributeNS(null, "propagateResolutionExceptions")));
        }

        final List<Element> attributeDependencyElements =
                ElementSupport.getChildElements(config, InputAttributeDefinitionParser.ELEMENT_NAME);
        final List<Element> dataConnectorDependencyElements =
                ElementSupport.getChildElements(config, InputDataConnectorParser.ELEMENT_NAME);       
        if ((null != attributeDependencyElements && !attributeDependencyElements.isEmpty()) ||
            (null != dataConnectorDependencyElements && !dataConnectorDependencyElements.isEmpty())) {
            if (failOnDependencies()) {
                log.error("{} Dependencies are not allowed.", getLogPrefix());
                throw new BeanCreationException(getLogPrefix() + " has meaningless Dependencies statements");
            }
            if (warnOnDependencies()) {
                log.warn("{} Dependencies are not allowed.", getLogPrefix());
            }
        }
        builder.addPropertyValue("attributeDependencies", 
                SpringSupport.parseCustomElements(attributeDependencyElements, parserContext, builder));
        builder.addPropertyValue("dataConnectorDependencies", 
                SpringSupport.parseCustomElements(dataConnectorDependencyElements, parserContext, builder));
    }
    
    /**
     * Get the effective activation condition to inject.
     * 
     * @param config configuration element
     * 
     * @return condition bean definition builder, or null
     */
    @Nullable protected BeanDefinitionBuilder getActivationCondition(@Nonnull final Element config) {
        
        BeanDefinitionBuilder rpBuilder = null;
        BeanDefinitionBuilder phasesBuilder = null;
        
        final Attr relyingPartiesAttr = config.getAttributeNodeNS(null, "relyingParties");
        final Attr excludeRelyingPartiesAttr = config.getAttributeNodeNS(null, "excludeRelyingParties");
        if (relyingPartiesAttr!=null) {
            if (relyingPartyPredicateClass == null) {
                log.warn("Ignoring relyingParties setting due to class instantiation failure at startup");
                return null;
            }
            if (excludeRelyingPartiesAttr!=null) {
                log.warn("excludeRelyingParties ignored, using relyingParties");
            }
            assert relyingPartyPredicateClass != null;
            rpBuilder = BeanDefinitionBuilder.genericBeanDefinition(relyingPartyPredicateClass);
            if (relyingPartyPredicateFactoryMethod != null) {
                rpBuilder.setFactoryMethod(relyingPartyPredicateFactoryMethod);
            }
            rpBuilder.addConstructorArgValue(
                    SpringSupport.getAttributeValueAsList(relyingPartiesAttr));
        } else if (excludeRelyingPartiesAttr != null) {
            if (relyingPartyPredicateClass == null) {
                log.warn("Ignoring excludeRelyingParties setting due to class instantiation failure at startup");
                return null;
            }
            assert relyingPartyPredicateClass != null;
            final BeanDefinitionBuilder unnegated =
                    BeanDefinitionBuilder.genericBeanDefinition(relyingPartyPredicateClass);
            if (relyingPartyPredicateFactoryMethod != null) {
                unnegated.setFactoryMethod(relyingPartyPredicateFactoryMethod);
            }
            unnegated.addConstructorArgValue(
                    SpringSupport.getAttributeValueAsList(excludeRelyingPartiesAttr));
            rpBuilder = BeanDefinitionBuilder.genericBeanDefinition(PredicateSupport.class);
            rpBuilder.setFactoryMethod("not");
            rpBuilder.addConstructorArgValue(unnegated.getBeanDefinition());
        }
        
        final Attr resolutionPhasesAttr = config.getAttributeNodeNS(null, "resolutionPhases");
        final Attr excludeResolutionPhasesAttr = config.getAttributeNodeNS(null, "excludeResolutionPhases");
        if (resolutionPhasesAttr!=null) {
            if (excludeResolutionPhasesAttr!=null) {
                log.warn("excludeResolutionPhases ignored, using resolutionPhases");
            }
            phasesBuilder = BeanDefinitionBuilder.genericBeanDefinition(ResolutionLabelPredicate.class);
            phasesBuilder.setFactoryMethod("byList");
            phasesBuilder.addConstructorArgValue(SpringSupport.getAttributeValueAsList(resolutionPhasesAttr));
        } else if (excludeResolutionPhasesAttr!=null) {
            final BeanDefinitionBuilder unnegated =
                    BeanDefinitionBuilder.genericBeanDefinition(ResolutionLabelPredicate.class);
            unnegated.setFactoryMethod("byList");
            unnegated.addConstructorArgValue(SpringSupport.getAttributeValueAsList(excludeResolutionPhasesAttr));
            phasesBuilder = BeanDefinitionBuilder.genericBeanDefinition(PredicateSupport.class);
            phasesBuilder.setFactoryMethod("not");
            phasesBuilder.addConstructorArgValue(unnegated.getBeanDefinition());
        }

        if (rpBuilder != null && phasesBuilder != null) {
            final BeanDefinitionBuilder andBuilder =
                    BeanDefinitionBuilder.genericBeanDefinition(PredicateSupport.class);
            andBuilder.setFactoryMethod("and");
            andBuilder.addConstructorArgValue(rpBuilder.getBeanDefinition());
            andBuilder.addConstructorArgValue(phasesBuilder.getBeanDefinition());
            return andBuilder;
        } else if (rpBuilder != null) {
            return rpBuilder;
        } else if (phasesBuilder != null) {
            return phasesBuilder;
        }
        
        return null;
    }
// Checkstyle: CyclomaticComplexity|MethodLength ON
    
    /** Controls parsing of Dependencies. 
     * 
     * If it is considered an invalid configuration for this resolver to have Dependency statements, return true. 
     * The surrounding logic will fail the parse.
     * @return false - by default.
     */
    protected boolean failOnDependencies() {
        return false;
    }

    /** Controls parsing of Dependencies. 
     * 
     * If it is considered an invalid configuration for this resolver to have Dependency statements, return true. 
     * The surrounding logic will issue warning.
     * @return false - by default.
     */
    protected boolean warnOnDependencies() {
        return false;
    }

    /**
     * Return a string which is to be prepended to all log messages.
     * 
     * This is always overridden by upper parsers, but to leave this abstract would break compatibility
     * 
     * @return a basic prefix.
     */
    @Nonnull @NotEmpty protected String getLogPrefix() {
        final StringBuilder builder = new StringBuilder("Unknown Plugin '").append(getDefinitionId()).append("':");
        final String result = builder.toString();
        assert result != null;
        return result;
    }

}