/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.ad.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.spring.ad.BaseAttributeDefinitionParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.idp.saml.attribute.resolver.impl.SAML1NameIdentifierAttributeDefinition;
import net.shibboleth.shared.primitive.DeprecationSupport;
import net.shibboleth.shared.primitive.DeprecationSupport.ObjectType;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/** Spring bean definition parser for SAML 1 NameIdentifier attribute definitions. */
@SuppressWarnings("removal")
public class SAML1NameIdentifierAttributeDefinitionParser extends BaseAttributeDefinitionParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "SAML1NameIdentifier");

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SAML1NameIdentifierAttributeDefinitionParser.class);

    /** {@inheritDoc} */
    @Override
    @Nullable protected Class<SAML1NameIdentifierAttributeDefinition> getBeanClass(@Nullable final Element element) {
        return SAML1NameIdentifierAttributeDefinition.class;
    }

    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {        
        DeprecationSupport.atRisk(ObjectType.XSITYPE, "SAML1NameIdentifier",
                parserContext.getReaderContext().getResource().getDescription());
        
        super.doParse(config, parserContext, builder);

        final String nameIdQualifier = StringSupport.trimOrNull(config.getAttributeNS(null, "nameIdQualifier"));
        builder.addPropertyValue("nameIdQualifier", nameIdQualifier);

        if (config.hasAttributeNS(null, "nameIdFormat")) {
            final String nameIdFormat = StringSupport.trimOrNull(config.getAttributeNS(null, "nameIdFormat"));
            log.debug("{} nameIdFormat '{}', nameIdQualifier '{}'", getLogPrefix(), nameIdFormat, nameIdQualifier);
            builder.addPropertyValue("nameIdFormat", nameIdFormat);
        } else {
            log.debug("{} nameIdQualifier '{}'", getLogPrefix(), nameIdQualifier);
        }
    }

}