/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.ad.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.ad.impl.ContextDerivedAttributeDefinition;
import net.shibboleth.idp.attribute.resolver.spring.ad.BaseAttributeDefinitionParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;

/** Spring Bean Definition Parser for attribute definitions derived from the Principal. */
public class SubjectDerivedAttributeDefinitionParser extends BaseAttributeDefinitionParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "SubjectDerivedAttribute");

    /** Class for sourcing values from Subject(s). */
    @Nullable private Class<? extends Function<?,?>> subjectDerivedClass;

    /** Class for sourcing values from Principal(s). */
    @Nullable private Class<? extends Function<?,?>> principalDerivedClass;

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SubjectDerivedAttributeDefinitionParser.class);

    /** Constructor. */
    @SuppressWarnings("unchecked")
    public SubjectDerivedAttributeDefinitionParser() {
        try {
            String className =
                    getCustomProperty(getClass().getName() + ".SubjectDerivedAttributeValuesFunction.class", null);
            if (className != null) {
                subjectDerivedClass = (Class<? extends Function<?, ?>>) Class.forName(className);
            } else {
                throw new ClassNotFoundException();
            }
            className =
                    getCustomProperty(getClass().getName() + ".PrincipalDerivedValuesFunction.class", null);
            if (className != null) {
                principalDerivedClass = (Class<? extends Function<?, ?>>) Class.forName(className);
            } else {
                throw new ClassNotFoundException();
            }
        } catch (final ClassNotFoundException | ClassCastException e) {
            log.error("Unable to load classes to support instantiation of this plugin type");
        }
    }
    
    /** {@inheritDoc} */
    @Override
    @Nullable protected Class<ContextDerivedAttributeDefinition> getBeanClass(@Nonnull final Element element) {
        return ContextDerivedAttributeDefinition.class;
    }

    /**
     * {@inheritDoc}.
     * 
     * We inject an inferred function to derive the attributes from the Subject.
     * 
     * <p>
     * If 'principalAttributeName' we also inject an inferred function to derive the attributes from a Principal. If
     * 'attributeValueFunctionRef' the user has provided the function. {@link ContextDerivedAttributeDefinitionParser}
     * handles the case when the user injects the top level function.
     * </p>
     */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        
        if (subjectDerivedClass == null) {
            throw new BeanCreationException("Unable to load class for subject-derived attribute function.");
        }
        
        super.doParse(config, parserContext, builder);
        final String attributeName = StringSupport.trimOrNull(config.getAttributeNS(null, "principalAttributeName"));
        final String functionRef = StringSupport.trimOrNull(config.getAttributeNS(null, "attributeValuesFunctionRef"));
        
        assert subjectDerivedClass != null;
        final BeanDefinitionBuilder contextFunctionBuilder =
                BeanDefinitionBuilder.genericBeanDefinition(subjectDerivedClass);
        contextFunctionBuilder.setInitMethodName("initialize");
        contextFunctionBuilder.setDestroyMethodName("destroy");
        contextFunctionBuilder.addPropertyValue("id", getDefinitionId());

        if (config.hasAttributeNS(null, "forCanonicalization")) {
            contextFunctionBuilder.addPropertyValue("forCanonicalization",
                    SpringSupport.getStringValueAsBoolean(config.getAttributeNS(null, "forCanonicalization")));
        }

        if (null != attributeName) {
            if (null != functionRef) {
                log.warn("{} only one of \"principalAttributeName\" or \"attributeValuesFunctionRef\""
                        + " should be provided. \"attributeValuesFunctionRef\" ignored", getLogPrefix());
            }
            
            if (principalDerivedClass == null) {
                throw new BeanCreationException("Unable to load class for principal-derived attribute function.");
            }
            
            assert principalDerivedClass != null;
            final BeanDefinitionBuilder principalValuesFunctionBuilder =
                    BeanDefinitionBuilder.genericBeanDefinition(principalDerivedClass);
            principalValuesFunctionBuilder.addPropertyValue("attributeName", attributeName);
            contextFunctionBuilder.addPropertyValue("attributeValuesFunction",
                    principalValuesFunctionBuilder.getBeanDefinition());
        } else if (null != functionRef) {
            contextFunctionBuilder.addPropertyReference("attributeValuesFunction", functionRef);
        } else {
            log.error("{} one of \"principalAttributeName\" or \"attributeValuesFunctionRef\" should be supplied."
                    + " should be provided.", getLogPrefix());
            throw new BeanCreationException("Misconfigured PrincipalDerivedAttribute.");
        }
        builder.addPropertyValue("attributeValuesFunction", contextFunctionBuilder.getBeanDefinition());
    }

    /** {@inheritDoc} */
    @Override protected boolean failOnDependencies() {
        return true;
    }
    
}