/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.ad.mapped.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.ad.mapped.impl.SourceValue;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.custom.AbstractCustomBeanDefinitionParser;
import net.shibboleth.shared.spring.util.SpringSupport;

/** Bean definition parser for a {@link SourceValue}. */
public class SourceValueParser extends AbstractCustomBeanDefinitionParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME_RESOLVER =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "SourceValue");

    /** Class logger. */
    @Nonnull private Logger log = LoggerFactory.getLogger(SourceValueParser.class);

    /** {@inheritDoc} */
    @Override
    @Nullable protected Class<SourceValue> getBeanClass(@Nullable final Element element) {
        return SourceValue.class;
    }

    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(config, parserContext, builder);

        builder.setInitMethodName("initialize");
        builder.setDestroyMethodName("destroy");
        
        final String value = config.getTextContent();
        builder.addPropertyValue("value", value);

        String caseSensitive = null;
        if (config.hasAttributeNS(null, "caseSensitive")) {
            caseSensitive = StringSupport.trimOrNull(config.getAttributeNS(null, "caseSensitive"));
            builder.addPropertyValue("caseSensitive", SpringSupport.getStringValueAsBoolean(caseSensitive));
        }

        String partialMatch = null;
        if (config.hasAttributeNS(null, "partialMatch")) {
            partialMatch = StringSupport.trimOrNull(config.getAttributeNS(null, "partialMatch"));
            builder.addPropertyValue("partialMatch", SpringSupport.getStringValueAsBoolean(partialMatch));
        }

        log.debug("SourceValue value: {}, caseSensitive: {}, partialMatch: {}", value, caseSensitive, partialMatch);

    }

    /** {@inheritDoc} */
    @Override protected boolean shouldGenerateId() {
        return true;
    }

}