/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.dc;

import java.util.List;

import javax.annotation.Nonnull;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.spring.BaseResolverPluginParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.ElementSupport;

/**
 * Base spring bean definition parser for data connectors. DataConnector implementations should provide a custom
 * BeanDefinitionParser by extending this class and overriding the
 * {@link #doParse(Element, ParserContext, BeanDefinitionBuilder)} method to parse any additional attributes or elements
 * it requires. Standard attributes and elements defined by the ResolutionPlugIn and DataConnector schemas will
 * automatically attempt to be parsed.
 */
public abstract class AbstractDataConnectorParser extends BaseResolverPluginParser {

    /** Element name. */
    @Nonnull public static final QName ELEMENT_NAME = new QName(AttributeResolverNamespaceHandler.NAMESPACE,
            "DataConnector");

    /** Delay in retrying failed connector. */
    @Nonnull @NotEmpty public static final String ATTR_NORETRYDELAY = "noRetryDelay";

    /**
     * Which attributes to export.
     */
    @Nonnull @NotEmpty public static final String ATTR_EXPORT_NAMES = "exportAttributes";
    
    /**
     * Failfast LDAP, Realtional, Stored.
     */
    @Nonnull @NotEmpty public static final String ATTR_FAIL_FAST = "failFastInitialize";

    /** Failfast default. */
    @Nonnull public static final Boolean FAIL_FAST_DEFAULT = false;

    /** Failover data connector attribute name. */
    @Nonnull public static final QName FAILOVER_DATA_CONNECTOR_ELEMENT_NAME = new QName(
            AttributeResolverNamespaceHandler.NAMESPACE, "FailoverDataConnector");

    /** Log4j logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractDataConnectorParser.class);

    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(config, parserContext, builder);

        final List<Element> failoverConnector =
                ElementSupport.getChildElements(config, FAILOVER_DATA_CONNECTOR_ELEMENT_NAME);

        if (failoverConnector != null && !failoverConnector.isEmpty()) {
            if (failoverConnector.size() > 1) {
                 log.warn("{} More than one failover data connector specified, taking the first", 
                        getLogPrefix());                
            }
            final String connectorId = StringSupport.trimOrNull(failoverConnector.get(0).getAttributeNS(null, "ref"));
            log.debug("{} Setting the following failover data connector dependencies: {}", getLogPrefix(), connectorId);
            builder.addPropertyValue("failoverDataConnectorId", connectorId);
        }

        if (config.hasAttributeNS(null, ATTR_NORETRYDELAY)) {
            builder.addPropertyValue("noRetryDelay",
                    StringSupport.trimOrNull(config.getAttributeNS(null, ATTR_NORETRYDELAY)));
        }

        final Attr export = config.getAttributeNodeNS(null, ATTR_EXPORT_NAMES);
        if (export!=null) {
            builder.addPropertyValue("exportAttributes", SpringSupport.getAttributeValueAsList(export));
        }

    }

    /**
     * Return a string which is to be prepended to all log messages.
     * 
     * @return "Data Connector: '&lt;definitionID&gt;' :"
     */
    @Override @Nonnull @NotEmpty protected String getLogPrefix() {
        final StringBuilder builder = new StringBuilder("Data Connector '").append(getDefinitionId()).append("':");
        final String result = builder.toString();
        assert result != null;
        return result;
    }
}
