/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.dc.http.impl;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.opensaml.security.httpclient.HttpClientSecurityParameters;
import org.opensaml.spring.credential.BasicX509CredentialFactoryBean;
import org.opensaml.spring.httpclient.HttpClientSecurityParametersMergingFactoryBean;
import org.opensaml.spring.trust.StaticExplicitKeyFactoryBean;
import org.opensaml.spring.trust.StaticPKIXFactoryBean;
import org.slf4j.Logger;
import org.springframework.beans.BeanMetadataElement;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.RuntimeBeanReference;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.ManagedList;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Attr;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.dc.http.impl.HTTPDataConnector;
import net.shibboleth.idp.attribute.resolver.dc.http.impl.ScriptedResponseMappingStrategy;
import net.shibboleth.idp.attribute.resolver.dc.http.impl.TemplatedBodyBuilder;
import net.shibboleth.idp.attribute.resolver.dc.http.impl.TemplatedURLBuilder;
import net.shibboleth.idp.attribute.resolver.spring.dc.AbstractDataConnectorParser;
import net.shibboleth.idp.attribute.resolver.spring.dc.impl.CacheConfigParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;
import net.shibboleth.shared.xml.ElementSupport;
import net.shibboleth.spring.ScriptTypeBeanParser;

/** Bean definition Parser for a {@link HTTPDataConnector}. */
public class HTTPDataConnectorParser extends AbstractDataConnectorParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "HTTP");

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(HTTPDataConnectorParser.class);

    /** {@inheritDoc} */
    @Override
    @Nullable protected Class<HTTPDataConnector> getBeanClass(@Nonnull final Element element) {
        return HTTPDataConnector.class;
    }
    
// Checkstyle: CyclomaticComplexity|MethodLength OFF
    /** {@inheritDoc} */
    @Override protected void doParse(@Nonnull final Element config, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {

        super.doParse(config, parserContext, builder);
        log.debug("{} Parsing custom configuration {}", getLogPrefix(), config);

        if (config.hasAttributeNS(null, ATTR_FAIL_FAST)) {
            // LDAP, Relational & HTTP only, limited in the schema
            builder.addPropertyValue("failFastInitialize",
                    StringSupport.trimOrNull(config.getAttributeNS(null, ATTR_FAIL_FAST)));
        } else {
            builder.addPropertyValue("failFastInitialize", FAIL_FAST_DEFAULT);
        }

        final V2Parser v2Parser = new V2Parser(config, getLogPrefix());

        final String httpClientID = v2Parser.getBeanHttpClientID();
        if (httpClientID != null) {
            builder.addPropertyReference("httpClient", httpClientID);
        }
        
        final BeanDefinition httpClientSecurityParameters = buildHttpClientSecurityParameters(
                v2Parser.buildHttpClientSecurityParams(config.getAttributeNS(null, "id")),
                StringSupport.trimOrNull(config.getAttributeNS(null, "httpClientSecurityParametersRef")),
                parserContext);
        builder.addPropertyValue("httpClientSecurityParameters", httpClientSecurityParameters);

        final String searchBuilderID = v2Parser.getBeanSearchBuilderID();
        if (searchBuilderID != null) {
            builder.addPropertyReference("executableSearchBuilder", searchBuilderID);
        } else {
            BeanDefinition def = v2Parser.createBodyTemplateBuilder(httpClientSecurityParameters);
            if (def != null) {
                builder.addPropertyValue("executableSearchBuilder", def);
            } else {
                def = v2Parser.createURLTemplateBuilder(httpClientSecurityParameters);
                if (def != null) {
                    builder.addPropertyValue("executableSearchBuilder", def);
                }
            }
        }

        final String mappingStrategyID = v2Parser.getBeanMappingStrategyID();
        if (mappingStrategyID != null) {
            builder.addPropertyReference("mappingStrategy", mappingStrategyID);
        } else {
            final BeanDefinition def = v2Parser.createMappingStrategy(config.getAttributeNS(null, "id"));
            if (def != null) {
                builder.addPropertyValue("mappingStrategy", def);
            }
        }
        
        final String validatorID = v2Parser.getBeanValidatorID();
        if (validatorID != null) {
            builder.addPropertyReference("validator", validatorID);
        }

        final String resultCacheBeanID = CacheConfigParser.getBeanResultCacheID(config);
        
        if (null != resultCacheBeanID) {
           builder.addPropertyReference("resultsCache", resultCacheBeanID);
        } else {
            builder.addPropertyValue("resultsCache", v2Parser.createCache(parserContext));
        }

        builder.setInitMethodName("initialize");
        builder.setDestroyMethodName("destroy");
    }
// Checkstyle: CyclomaticComplexity|MethodLength ON
    
    /**
     * Build the definition of the HttpClientSecurityParameters as a factory bean which merges the inline params data 
     * and parameters ref inputs.
     * 
     * @param inlineParams the bean definition for the inline params data such as TLS TrustEngine and client Credential
     * @param parametersRef the security parameters bean reference
     * @param parserContext context
     * @return the bean definition with the parameters.
     */
    @Nonnull protected static BeanDefinition buildHttpClientSecurityParameters(
            @Nullable final BeanDefinition inlineParams, @Nullable final String parametersRef,
            @Nonnull final ParserContext parserContext) {
        
        final BeanDefinitionBuilder factoryBuilder =
                BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParametersMergingFactoryBean.class);
        
        final List<BeanMetadataElement> factoryInputs = new ManagedList<>(2);

        // First order-of-precedence
        if (inlineParams != null)  {
            factoryInputs.add(inlineParams);
        }

        // Second order-of-precedence
        if (parametersRef != null) {
            factoryInputs.add(new RuntimeBeanReference(parametersRef));
        }

        factoryBuilder.addPropertyValue("parameters", factoryInputs);
        
        return factoryBuilder.getBeanDefinition();
    }
    

    /**
     * Utility class for parsing v2 schema configuration.
     * 
     */
    protected static class V2Parser {

        /** Base XML element. */
        @Nonnull private final Element configElement;

        /** Class logger. */
        @Nonnull private final Logger log = LoggerFactory.getLogger(V2Parser.class);

        /** Parent parser's log prefix.*/
        @Nonnull @NotEmpty private final String logPrefix;

        /**
         * Creates a new V2Parser with the supplied element.
         * 
         * @param config HTTP DataConnector element
         * @param prefix the parent parser's log prefix.
         */
        public V2Parser(@Nonnull final Element config,  @Nonnull @NotEmpty final String prefix) {
            Constraint.isNotNull(config, "HTTP DataConnector element cannot be null");
            configElement = config;
            logPrefix = prefix;
        }

        /**
         * Get the bean ID of an externally defined HttpClient.
         * 
         * @return HttpClient bean ID
         */
        @Nullable public String getBeanHttpClientID() {
            return AttributeSupport.getAttributeValue(configElement, new QName("httpClientRef"));
        }

        /**
         * Get the bean ID of an externally defined search builder.
         * 
         * @return search builder bean ID
         */
        @Nullable public String getBeanSearchBuilderID() {
            return AttributeSupport.getAttributeValue(configElement, null, "executableSearchBuilderRef");
        }
        
        /**
         * Create the definition of the GET search builder.
         * 
         * @param httpClientSecurityParams instance of {@link HttpClientSecurityParameters} to inject
         * 
         * @return the bean definition for the search builder
         */
        @Nullable public BeanDefinition createURLTemplateBuilder(
                @Nullable final BeanDefinition httpClientSecurityParams) {
            
            final List<Element> urlTemplates = ElementSupport.getChildElements(configElement, 
                    new QName(AttributeResolverNamespaceHandler.NAMESPACE, "URLTemplate"));
            if (urlTemplates.size() == 0) {
                return null;
            }
            
            final BeanDefinitionBuilder templateBuilder =
                    BeanDefinitionBuilder.genericBeanDefinition(TemplatedURLBuilder.class);
            templateBuilder.setInitMethodName("initialize");
            templateBuilder.setDestroyMethodName("destroy");

            String velocityEngineRef = StringSupport.trimOrNull(configElement.getAttributeNS(null, "templateEngine"));
            if (null == velocityEngineRef) {
                velocityEngineRef = "shibboleth.VelocityEngine";
            }
            templateBuilder.addPropertyReference("velocityEngine", velocityEngineRef);

            // This is duplication but allows the built-in builder to access the parameters if desired.
            if (httpClientSecurityParams != null) {
                templateBuilder.addPropertyValue("httpClientSecurityParameters", httpClientSecurityParams);
            }
            
            if (urlTemplates.size() > 1) {
                log.warn("{} A maximum of 1 <URLTemplate> should be specified; the first one has been used",
                        getLogPrefix());
            }
            
            String url = null;
            if (!urlTemplates.isEmpty()) {
                url = urlTemplates.get(0).getTextContent();
            }
            templateBuilder.addPropertyValue("templateText", url);

            final String customRef =
                    StringSupport.trimOrNull(urlTemplates.get(0).getAttributeNS(null, "customObjectRef"));
            if (null != customRef) {
                templateBuilder.addPropertyReference("customObject", customRef);
            }
            
            final String headerMapRef = StringSupport.trimOrNull(configElement.getAttributeNS(null, "headerMapRef"));
            if (headerMapRef != null) {
                templateBuilder.addPropertyReference("headers", headerMapRef);
            }

            return templateBuilder.getBeanDefinition();
        }

// Checkstyle: CyclomaticComplexity|MethodLength OFF
        /**
         * Create the definition of the POST search builder.
         * 
         * @param httpClientSecurityParams instance of {@link HttpClientSecurityParameters} to inject
         * 
         * @return the bean definition for the search builder, or null
         */
        @Nullable public BeanDefinition createBodyTemplateBuilder(
                @Nullable final BeanDefinition httpClientSecurityParams) {
            
            final List<Element> urlTemplates = ElementSupport.getChildElements(configElement, 
                    new QName(AttributeResolverNamespaceHandler.NAMESPACE, "URLTemplate"));
            final List<Element> bodyTemplates = ElementSupport.getChildElements(configElement, 
                    new QName(AttributeResolverNamespaceHandler.NAMESPACE, "BodyTemplate"));
            if (urlTemplates.size() == 0 || bodyTemplates.size() == 0) {
                return null;
            }
            final List<Element> cacheKeyTemplates = ElementSupport.getChildElements(configElement, 
                    new QName(AttributeResolverNamespaceHandler.NAMESPACE, "CacheKeyTemplate"));
            
            final BeanDefinitionBuilder templateBuilder =
                    BeanDefinitionBuilder.genericBeanDefinition(TemplatedBodyBuilder.class);
            templateBuilder.setInitMethodName("initialize");
            templateBuilder.setDestroyMethodName("destroy");

            String velocityEngineRef = StringSupport.trimOrNull(configElement.getAttributeNS(null, "templateEngine"));
            if (null == velocityEngineRef) {
                velocityEngineRef = "shibboleth.VelocityEngine";
            }
            templateBuilder.addPropertyReference("velocityEngine", velocityEngineRef);

            // This is duplication but allows the built-in builder to access the parameters if desired.
            if (httpClientSecurityParams != null) {
                templateBuilder.addPropertyValue("httpClientSecurityParameters", httpClientSecurityParams);
            }
            
            if (urlTemplates.size() > 1) {
                log.warn("{} A maximum of 1 <URLTemplate> should be specified; the first one has been used",
                        getLogPrefix());
            }
            
            if (bodyTemplates.size() > 1) {
                log.warn("{} A maximum of 1 <BodyTemplate> should be specified; the first one has been used",
                        getLogPrefix());
            }
            
            if (cacheKeyTemplates.size() > 1) {
                log.warn("{} A maximum of 1 <CacheKeyTemplate> should be specified; the first one has been used",
                        getLogPrefix());
            }
            
            final Element urlTemplate = urlTemplates.get(0);
            final Element bodyTemplate = bodyTemplates.get(0);
            if (urlTemplate.hasAttributeNS(null, "customObjectRef")) {
                templateBuilder.addPropertyReference("customObject",
                        AttributeSupport.ensureAttributeValue(urlTemplate, null, "customObjectRef"));
                if (bodyTemplate.hasAttributeNS(null, "customObjectRef")) {
                    log.warn("{} Ignored <BodyTemplate> customObjectRef in favor of <URLTemplate> customObjectRef",
                            getLogPrefix());
                }
            } else if (bodyTemplate.hasAttributeNS(null, "customObjectRef")) {
                templateBuilder.addPropertyReference("customObject",
                        AttributeSupport.ensureAttributeValue(bodyTemplate, null, "customObjectRef"));
            }
            
            templateBuilder.addPropertyValue("uRLTemplateText", urlTemplate.getTextContent());
            
            templateBuilder.addPropertyValue("bodyTemplateText", bodyTemplate.getTextContent());
            if (bodyTemplate.hasAttributeNS(null, "MIMEType")) {
                templateBuilder.addPropertyValue("mIMEType", bodyTemplate.getAttributeNS(null, "MIMEType"));
            }
            if (bodyTemplate.hasAttributeNS(null, "charset")) {
                templateBuilder.addPropertyValue("characterSet", bodyTemplate.getAttributeNS(null, "charset"));
            }
            if (!cacheKeyTemplates.isEmpty()) {
                templateBuilder.addPropertyValue("cacheKeyTemplateText", cacheKeyTemplates.get(0).getTextContent());
            }
            final String headerMapRef = StringSupport.trimOrNull(configElement.getAttributeNS(null, "headerMapRef"));
            if (headerMapRef != null) {
                templateBuilder.addPropertyReference("headers", headerMapRef);
            }

            return templateBuilder.getBeanDefinition();
        }
// Checkstyle: CyclomaticComplexity|MethodLength ON
        
        /**
         * Get the bean ID of an externally defined mapping strategy.
         * 
         * @return mapping strategy bean ID
         */
        @Nullable public String getBeanMappingStrategyID() {
            return AttributeSupport.getAttributeValue(configElement, null, "mappingStrategyRef");
        }

        /**
         * Create the scripted result mapping strategy.
         * 
         * @param id the ID of the 
         * 
         * @return mapping strategy
         */
        @Nullable public BeanDefinition createMappingStrategy(@Nullable final String id) {

            final List<Element> mappings = ElementSupport.getChildElements(configElement, 
                    new QName(AttributeResolverNamespaceHandler.NAMESPACE, "ResponseMapping"));
    
            if (mappings.size() > 1) {
                log.warn("{} A maximum of 1 <ResponseMapping> should be specified; the first one has been used",
                        getLogPrefix());
            }
            if (mappings.isEmpty()) {
                log.error("{} No <ResponseMapping> provided", getLogPrefix());
                throw new BeanCreationException("No <ResponseMapping> provided");
            }
            final Element map = mappings.get(0);
            assert map != null;
            final BeanDefinitionBuilder mapper =
                    ScriptTypeBeanParser.parseScriptType(ScriptedResponseMappingStrategy.class, map);
            if (id != null) {
                mapper.addPropertyValue("logPrefix", id + ':');
            }
            
            final String maxLength = StringSupport.trimOrNull(configElement.getAttributeNS(null, "maxLength"));
            if (maxLength != null) {
                mapper.addPropertyValue("maxLength", maxLength);
            }
            
            final Attr acceptStatusesAttr = configElement.getAttributeNodeNS(null, "acceptStatuses");
            if (acceptStatusesAttr!=null) {
                mapper.addPropertyValue("acceptStatuses",
                        SpringSupport.getAttributeValueAsList(acceptStatusesAttr));
            }

            final Attr acceptTypesAttr = configElement.getAttributeNodeNS(null, "acceptTypes");
            
            if (acceptTypesAttr!=null) {
                mapper.addPropertyValue("acceptTypes",
                        SpringSupport.getAttributeValueAsList(acceptTypesAttr));
            }

            return mapper.getBeanDefinition();
        }
        
        /**
         * Get the bean ID of an externally defined validator.
         * 
         * @return validator bean ID
         */
        @Nullable public String getBeanValidatorID() {
            return AttributeSupport.getAttributeValue(configElement, null, "validatorRef");
        }
        
        /**
         * Create the results cache. See {@link CacheConfigParser}.
         * 
         * @param parserContext bean parser context
         * 
         * @return results cache
         */
        @Nullable public BeanDefinition createCache(@Nonnull final ParserContext parserContext) {
            final CacheConfigParser parser = new CacheConfigParser(configElement);
            return parser.createCache();
        }
        
        /**
         * Check for manually supplied settings and build {@link HttpClientSecurityParameters} instance.
         * 
         * @param id connector ID
         * 
         * @return a bean definition if applicable
         */
        @Nullable public BeanDefinition buildHttpClientSecurityParams(@Nullable final String id) {
            BeanDefinitionBuilder builder = null;
            final String serverCertificate =
                    StringSupport.trimOrNull(configElement.getAttributeNS(null, "serverCertificate"));
            if (serverCertificate != null) {
                log.debug("Auto-configuring connector {} with a server certificate to authenticate", id);
                builder = BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParameters.class);
                final BeanDefinitionBuilder explicitTrust =
                        BeanDefinitionBuilder.genericBeanDefinition(StaticExplicitKeyFactoryBean.class);
                explicitTrust.addPropertyValue("certificates", serverCertificate);
                builder.addPropertyValue("tLSTrustEngine", explicitTrust.getBeanDefinition());
            }
            
            final String certificateAuthority =
                    StringSupport.trimOrNull(configElement.getAttributeNS(null, "certificateAuthority"));
            if (certificateAuthority != null) {
                if (builder != null) {
                    log.warn("Ignoring certificateAuthority on connector {}, superseded by serverCertificate", id);
                } else {
                    log.debug("Auto-configuring connector {} with a certificate authority to authenticate", id);
                    builder = BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParameters.class);
                    final BeanDefinitionBuilder pkixTrust =
                            BeanDefinitionBuilder.genericBeanDefinition(StaticPKIXFactoryBean.class);
                    // Relies on underlying enhanced TLSSocketFactory hostname verifier to kick in.
                    pkixTrust.addPropertyValue("checkNames", false);
                    pkixTrust.addPropertyValue("certificates", certificateAuthority);
                    builder.addPropertyValue("tLSTrustEngine", pkixTrust.getBeanDefinition());
                }
            }
            
            final String clientPrivateKey =
                    StringSupport.trimOrNull(configElement.getAttributeNS(null, "clientPrivateKey"));
            final String clientCertificate =
                    StringSupport.trimOrNull(configElement.getAttributeNS(null, "clientCertificate"));
            if (clientPrivateKey != null && clientCertificate != null) {
                log.debug("Auto-configuring connector {} with client TLS credential", id);
                if (builder == null) {
                    builder = BeanDefinitionBuilder.genericBeanDefinition(HttpClientSecurityParameters.class);
                }
                final BeanDefinitionBuilder credentialBuilder =
                        BeanDefinitionBuilder.genericBeanDefinition(BasicX509CredentialFactoryBean.class);
                credentialBuilder.addPropertyValue("privateKey", clientPrivateKey);
                credentialBuilder.addPropertyValue("certificates", clientCertificate);
                builder.addPropertyValue("clientTLSCredential", credentialBuilder.getBeanDefinition());
            }
            
            return builder != null ? builder.getBeanDefinition() : null;
        }
        
        /** The parent parser's log prefix.
         * @return the log prefix.
         */
        @Nonnull @NotEmpty private String getLogPrefix() {
            return logPrefix;
        }
    }
    
}
