/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.dc.impl;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.dc.impl.ContextDerivedDataConnector;
import net.shibboleth.idp.attribute.resolver.spring.dc.AbstractDataConnectorParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;

/** Spring Bean Definition Parser for {@link ContextDerivedDataConnector}. */
public class ContextDerivedDataConnectorParser extends AbstractDataConnectorParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "ContextDerived");

    /** {@inheritDoc} */
    @Override
    @Nullable protected Class<? extends AbstractDataConnector> getBeanClass(@Nonnull final Element element) {
        return ContextDerivedDataConnector.class;
    }

    /** {@inheritDoc} */
    @Override
    protected void doParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(element, parserContext, builder);
        final String functionRef = StringSupport.trimOrNull(element.getAttributeNS(null, "attributesFunctionRef"));

        if (null == functionRef) {
            throw new BeanCreationException(getLogPrefix() + "requires 'attributesFunctionRef'");
        }
        builder.addPropertyReference("attributesFunction", functionRef);

        final String noResultIsError =
                AttributeSupport.getAttributeValue(element, new QName("noResultIsError"));
        if (noResultIsError != null) {
            builder.addPropertyValue("noResultIsError", SpringSupport.getStringValueAsBoolean(noResultIsError));
        }
    }
    
    /** {@inheritDoc} */
    @Override protected boolean failOnDependencies() {
        return true;
    }

}