/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.dc.impl;

import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.slf4j.Logger;
import org.springframework.beans.factory.BeanCreationException;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.xml.ParserContext;
import org.w3c.dom.Element;

import net.shibboleth.idp.attribute.resolver.AbstractDataConnector;
import net.shibboleth.idp.attribute.resolver.dc.impl.ContextDerivedDataConnector;
import net.shibboleth.idp.attribute.resolver.spring.dc.AbstractDataConnectorParser;
import net.shibboleth.idp.attribute.resolver.spring.impl.AttributeResolverNamespaceHandler;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.spring.util.SpringSupport;
import net.shibboleth.shared.xml.AttributeSupport;

/** Spring Bean Definition Parser for {@link ContextDerivedDataConnector} with a predefined mapping function. 
 */
public class SubjectDataConnectorParser extends AbstractDataConnectorParser {

    /** Schema type name. */
    @Nonnull public static final QName TYPE_NAME =
            new QName(AttributeResolverNamespaceHandler.NAMESPACE, "Subject");

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(SubjectDataConnectorParser.class);

    /** Class for sourcing values from Subject(s). */
    @Nullable private Class<? extends Function<?,?>> subjectDerivedClass;
    
    /** Constructor. */
    @SuppressWarnings("unchecked")
    public SubjectDataConnectorParser() {
        try {
            final String className =
                    getCustomProperty(getClass().getName() + ".SubjectDerivedAttributesFunction.class", null);
            if (className != null) {
                subjectDerivedClass = (Class<? extends Function<?, ?>>) Class.forName(className);
            } else {
                throw new ClassNotFoundException();
            }
        } catch (final ClassNotFoundException|ClassCastException e) {
            log.error("Unable to load class to support instantiation of this plugin type");
        }
    }

    /** {@inheritDoc} */
    @Override
    @Nullable  protected Class<? extends AbstractDataConnector> getBeanClass(@Nonnull final Element element) {
        return ContextDerivedDataConnector.class;
    }

    /** {@inheritDoc} */
    @Override
    protected void doParse(@Nonnull final Element element, @Nonnull final ParserContext parserContext,
            @Nonnull final BeanDefinitionBuilder builder) {
        super.doParse(element, parserContext, builder);
        if (subjectDerivedClass == null) {
            throw new BeanCreationException("Unable to load class for subject-derived attribute function.");
        }

        // Auto-inject an instance of the deferred class type as the lookup function.
        
        assert subjectDerivedClass != null;
        final BeanDefinitionBuilder contextFunctionBuilder =
                BeanDefinitionBuilder.genericBeanDefinition(subjectDerivedClass);
        contextFunctionBuilder.setInitMethodName("initialize");
        contextFunctionBuilder.setDestroyMethodName("destroy");
        contextFunctionBuilder.addPropertyValue("id", getDefinitionId());

        if (element.hasAttributeNS(null, "forCanonicalization")) {
            contextFunctionBuilder.addPropertyValue("forCanonicalization",
                    SpringSupport.getStringValueAsBoolean(element.getAttributeNS(null, "forCanonicalization")));
        }
        
        builder.addPropertyValue("attributesFunction", contextFunctionBuilder.getBeanDefinition());
        
        final String noResultIsError =
                AttributeSupport.getAttributeValue(element, new QName("noResultIsError"));
        if (noResultIsError != null) {
            builder.addPropertyValue("noResultIsError", SpringSupport.getStringValueAsBoolean(noResultIsError));
        }
    }

    /** {@inheritDoc} */
    @Override protected boolean failOnDependencies() {
        return true;
    }

}