/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.impl;

import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.profile.context.ProfileRequestContext;

import net.shibboleth.idp.attribute.resolver.AbstractResolverPlugin;
import net.shibboleth.idp.attribute.resolver.ResolverAttributeDefinitionDependency;
import net.shibboleth.idp.attribute.resolver.ResolverDataConnectorDependency;
import net.shibboleth.shared.spring.factory.AbstractComponentAwareFactoryBean;

/**
 * A factory bean to collect the parameterization that goes onto a {@link AbstractResolverPlugin}.
 * 
 * @param <T> The exact type being deployed.
 */
public abstract class AbstractResolverPluginFactoryBean<T extends AbstractResolverPlugin<?>> extends
        AbstractComponentAwareFactoryBean<T> {

    /** The component Id. */
    @Nullable private String componentId;
    
    /** Plugin property "propagateResolutionExceptions". */
    @Nullable private Boolean propagateResolutionExceptions;

    /** Plugin property "activationCondition". */
    @Nullable private Predicate<ProfileRequestContext> activationCondition;

    /** Plugin property "attributeDependencies". */
    @Nullable private Set<ResolverAttributeDefinitionDependency> attributeDependencies;

    /** Plugin property "dataConnectorDependencies". */
    @Nullable private Set<ResolverDataConnectorDependency> dataConnectorDependencies;

    /** Get the component Id.
     * @return the id.
     */
    @Nullable public String getId() {
        return componentId;
    }
    
    /** Set the component Id.
     * @param id the id.
     */
    public void setId(@Nullable final String id) {
        componentId = id;
    }
    
    /**
     * Bean setter in support of {@link AbstractResolverPlugin#setPropagateResolutionExceptions(boolean)}.
     * 
     * @param propagate value to be set
     */
    public void setPropagateResolutionExceptions(final boolean propagate) {
        propagateResolutionExceptions = propagate;
    }

    /**
     * Bean getter in support of {@link AbstractResolverPlugin#setPropagateResolutionExceptions(boolean)}.
     * 
     * @return The value to be set
     */
    @Nullable public Boolean getPropagateResolutionExceptions() {
        return propagateResolutionExceptions;
    }

    /**
     * Bean setter in support of {@link AbstractResolverPlugin#setActivationCondition(Predicate)}.
     * 
     * @param pred what to set
     */
    public void setActivationCondition(@Nullable final Predicate<ProfileRequestContext> pred) {
        activationCondition = pred;
    }

    /**
     * Bean getter in support of {@link AbstractResolverPlugin#setActivationCondition(Predicate)}.
     * 
     * @return The value to be set
     */
    @Nullable public Predicate<ProfileRequestContext> getActivationCondition() {
        return activationCondition;
    }

    /**
     * Bean setter in support of {@link AbstractResolverPlugin#setAttributeDependencies(Set)}.
     * 
     * @param dependencies value to set
     */
    public void setAttributeDependencies(@Nullable final Set<ResolverAttributeDefinitionDependency> dependencies) {

        attributeDependencies = dependencies;
    }
    
    /**
     * Bean setter in support of {@link AbstractResolverPlugin#setDataConnectorDependencies(Set)}.
     * 
     * @param dependencies value to set
     */
    public void setDataConnectorDependencies(@Nullable final Set<ResolverDataConnectorDependency> dependencies) {

        dataConnectorDependencies = dependencies;
    }

    /**
     * Bean getter in support of {@link AbstractResolverPlugin#setAttributeDependencies(Set)}.
     * 
     * @return The value
     */
    @Nullable public Set<ResolverAttributeDefinitionDependency> getAttributeDependencies() {
        return attributeDependencies;
    }

    /**
     * Bean getter in support of {@link AbstractResolverPlugin#setDataConnectorDependencies(Set)}.
     * 
     * @return The value
     */
    @Nullable public Set<ResolverDataConnectorDependency> getDataConnectorDependencies() {
        return dataConnectorDependencies;
    }

    /** 
     * Set the locally defined values into the object under construction.
     * @param what the object being built.
     */
    protected void setValues(@Nonnull final T what) {
        final String id = getId();
        if (null != id) {
            what.setId(id);
        }
        
        final Predicate<ProfileRequestContext> condition = getActivationCondition();
        if (null != condition) {
            what.setActivationCondition(condition);
        }
        
        final Set<ResolverAttributeDefinitionDependency> depad = getAttributeDependencies(); 
        if (null != depad) {
            what.setAttributeDependencies(depad);
        }
        
        final Set<ResolverDataConnectorDependency> depdc = getDataConnectorDependencies(); 
        if (null != depdc) {
            what.setDataConnectorDependencies(depdc);
        }
        
        final Boolean flag = getPropagateResolutionExceptions();
        if (null != flag) {
            what.setPropagateResolutionExceptions(flag);
        }
    }
    
}