/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.attribute.resolver.spring.impl;

import java.util.Collection;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.springframework.context.ApplicationContext;

import net.shibboleth.idp.attribute.resolver.AttributeDefinition;
import net.shibboleth.idp.attribute.resolver.AttributeResolver;
import net.shibboleth.idp.attribute.resolver.DataConnector;
import net.shibboleth.idp.attribute.resolver.impl.AttributeResolverImpl;
import net.shibboleth.shared.component.AbstractIdentifiableInitializableComponent;
import net.shibboleth.shared.component.ComponentInitializationException;
import net.shibboleth.shared.logic.NonnullFunction;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.service.ServiceException;
import net.shibboleth.shared.spring.service.AbstractServiceableComponent;
import net.shibboleth.shared.spring.service.impl.SpringServiceableComponent;

/**
 * Strategy for summoning up an {@link AttributeResolverImpl} from a populated {@link ApplicationContext}. We do this by
 * finding all the configured {@link AttributeDefinition} and {@link DataConnector} beans and bunging them into the
 * Attribute Resolver which we then initialize.
 */
public class AttributeResolverServiceStrategy extends AbstractIdentifiableInitializableComponent implements
    NonnullFunction<ApplicationContext,AbstractServiceableComponent<AttributeResolver>> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AttributeResolverServiceStrategy.class);
    
    /** Whether to strip null attribute values. */
    private boolean stripNulls;

    /** Do we strip nulls from attribute values.
    * @return Returns whether to strip nulls from attribute values
    */
    public boolean isStripNulls() {
        return stripNulls;
    }

    /** 
     * Sets whether to strip nulls from attribute values.
     * @param doStripNulls what to set 
     */
    public void setStripNulls(final boolean doStripNulls) {
       stripNulls = doStripNulls;
    }

    /** {@inheritDoc} */
    @Nonnull public AbstractServiceableComponent<AttributeResolver> apply(
            @Nullable final ApplicationContext appContext) {

        if (appContext == null) {
            throw new ServiceException("ApplicationContext was null");
        }
        
        final Collection<AttributeDefinition> definitions =
                appContext.getBeansOfType(AttributeDefinition.class).values();

        final Collection<DataConnector> connectors = appContext.getBeansOfType(DataConnector.class).values();

        log.debug("Creating Attribute Resolver {} with {} Attribute Definition(s) and {} Data Connector(s)",
                getId(), definitions.size(), connectors.size());

        final AttributeResolverImpl resolver = new AttributeResolverImpl();
        resolver.setAttributeDefinitions(definitions);
        resolver.setDataConnectors(connectors);
        resolver.setId(ensureId());
        resolver.setStripNulls(isStripNulls());
        final SpringServiceableComponent<AttributeResolver> result;
        try {
            resolver.initialize();
            result = new SpringServiceableComponent<AttributeResolver>(resolver);
            result.setApplicationContext(appContext);
            result.setId(ensureId());
            result.initialize();
            return result;
        } catch (final ComponentInitializationException e) {
            throw new ServiceException("Unable to initialize attribute resolver for " + appContext.getDisplayName(), e);
        }
    }
    
}