/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.metadata;

import java.util.Locale;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.opensaml.saml.saml2.metadata.AttributeConsumingService;
import org.opensaml.saml.saml2.metadata.LocalizedName;
import org.slf4j.Logger;

import net.shibboleth.shared.annotation.constraint.NotLive;
import net.shibboleth.shared.annotation.constraint.Unmodifiable;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Class to contain a processed form of the {@link AttributeConsumingService} suitable for display purposes. */
public class ACSUIInfo {
    
    /** logger. */
    @Nonnull private static final Logger LOG = LoggerFactory.getLogger(ACSUIInfo.class);
    
    /** The Service Names as a map from locale to actual value.*/ 
    @Nonnull @Unmodifiable @NotLive private final Map<Locale, String> serviceNames;
    
    /** The Service Descriptions as a map from locale to actual value.*/ 
    @Nonnull @Unmodifiable @NotLive private final Map<Locale, String> serviceDescriptions;
       
    /** Warning check against a non localized String. */
    @Nonnull private final Predicate<LocalizedName> nullLanguageString = new Predicate<>() {
        public boolean test(final LocalizedName u) {
            if (u.getXMLLang() == null) {
                LOG.warn("<{}> with value {} in <AttributeConsumingService>" +
               " has no language associated, ignoring", u.getElementQName().getLocalPart(), u.getValue());
                return false;
            } 
            if (u.getValue() == null) {
                LOG.warn("Ignoring empty <{}> element in <AttributeConsumingService>",
                        u.getElementQName().getLocalPart(), u.getValue());
                return false;
            }
            return true;
        }
    };

    /**
     * Constructor.
     *
     * @param acs The OpenSaml AssertionConsumingService to convert.
     */
    public ACSUIInfo(@Nonnull final AttributeConsumingService acs) {
        
        serviceNames = acs.getNames().
                stream().
                filter(nullLanguageString).
                collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableMap(
                        serviceName -> Locale.forLanguageTag(serviceName.getXMLLang()), 
                        serviceName -> serviceName.getValue(),
                        CollectionSupport.warningMergeFunction("ServiceName", false)))).
                get();

        serviceDescriptions = acs.
                getDescriptions().
                stream().
                filter(nullLanguageString).
                collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableMap(
                        description -> Locale.forLanguageTag(description.getXMLLang()), 
                        description -> description.getValue(),
                        CollectionSupport.warningMergeFunction("ServiceDescription", false)))).
                get();
    }

    /** 
     * Get the Display Names as a map from locale to actual value.
     * 
     * @return the display names
     */
    @Nonnull @Unmodifiable @NotLive public Map<Locale, String> getServiceNames() {
        return serviceNames; 
    }
    
    /**
     * Return the descriptions as a map from locale to actual value.
     *
     * @return the descriptions names (if any)
     */
    @Nonnull @Unmodifiable @NotLive public Map<Locale, String> getServiceDescriptions() {
        return serviceDescriptions;
    }
    
}
