/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.xmlobject;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.xml.namespace.QName;

import org.opensaml.saml.common.SAMLObject;

import net.shibboleth.shared.annotation.constraint.NotEmpty;

/** XMLObject that represents a SAML attribute value whose value contains a scope attribute. */
public interface ScopedValue extends SAMLObject {

    /** Local name of the XSI type. */
    @Nonnull @NotEmpty static final String TYPE_LOCAL_NAME = "ScopedValue";

    /** QName of the XSI type. */
    // TODO fix these with constants when available
    @Nonnull static final QName TYPE_NAME =
            new QName("urn:mace:shibboleth:2.0:attribute:encoder", TYPE_LOCAL_NAME, "encoder");

    /**
     * Gets the name of the scope attribute.
     * 
     * @return name of the scope attribute
     */
    @Nullable String getScopeAttributeName();

    /**
     * Sets the name of the scope attribute.
     * 
     * @param attribute name of the scope attribute
     */
    void setScopeAttributeName(@Nullable final String attribute);

    /**
     * Gets the scope value.
     * 
     * @return scope value
     */
    @Nullable String getScope();

    /**
     * Sets the scope value.
     * 
     * @param scope scope value
     */
    void setScope(@Nullable final String scope);

    /**
     * Get the attribute value.
     * 
     * @return the attribute value.
     */
    @Nullable String getValue();

    /**
     * Set the attribute value.
     * 
     * @param newContent the new attribute value
     */
    void setValue(@Nullable final String newContent);
    
}