/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.xmlobject.impl;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;
import javax.xml.namespace.QName;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.io.AbstractXMLObjectUnmarshaller;
import org.opensaml.core.xml.io.UnmarshallingException;
import org.opensaml.xmlsec.signature.KeyInfo;
import org.slf4j.Logger;
import org.w3c.dom.Attr;

import net.shibboleth.idp.saml.xmlobject.KeyAuthority;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.xml.QNameSupport;

/** Unmarshaller for {@link KeyAuthority}. */
@ThreadSafe
public class KeyAuthorityUnmarshaller extends AbstractXMLObjectUnmarshaller {

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(KeyAuthorityUnmarshaller.class);

    /** {@inheritDoc} */
    protected void processAttribute(@Nonnull final XMLObject xmlObject, @Nonnull final Attr attribute)
            throws UnmarshallingException {
        final KeyAuthority authority = (KeyAuthority) xmlObject;

        if (attribute.getLocalName().equals(KeyAuthority.VERIFY_DEPTH_ATTRIB_NAME)) {
            authority.setVerifyDepth(Integer.valueOf(attribute.getValue()));
        } else {
            final QName attribQName = QNameSupport.getNodeQName(attribute);
            if (attribute.isId()) {
                authority.getUnknownAttributes().registerID(attribQName);
            }
            authority.getUnknownAttributes().put(attribQName, attribute.getValue());
        }
    }

    /** {@inheritDoc} */
    protected void processChildElement(@Nonnull final XMLObject parentXMLObject,
            @Nonnull final XMLObject childXMLObject) throws UnmarshallingException {
        final KeyAuthority authority = (KeyAuthority) parentXMLObject;

        if (childXMLObject instanceof KeyInfo) {
            authority.getKeyInfos().add((KeyInfo) childXMLObject);
        } else {
            super.processChildElement(parentXMLObject, childXMLObject);
        }
    }

}