/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.httpclient.resource;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import javax.annotation.Nonnull;

import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;

import org.apache.hc.client5.http.classic.HttpClient;
import org.apache.hc.core5.http.HttpResponse;
import org.apache.hc.core5.http.HttpStatus;
import org.slf4j.Logger;

import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import com.google.common.io.ByteStreams;

/**
 * A resource representing a file read from an HTTP(S) location. Every time the file is successfully read from the URL
 * location it is written to a backing file. If the file can not be read from the URL it is read from this backing file,
 * if available.
 * 
 */
public class FileBackedHTTPResource extends HTTPResource {

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(FileBackedHTTPResource.class);

    /** Backing resource file. */
    @Nonnull private final Resource backingResource;

    /**
     * Constructor.
     * 
     * @param backingFile the file to use as backing store
     * @param client the client we use to connect with.
     * @param url URL to the remote data
     * @throws IOException if the URL was badly formed
     */
    public FileBackedHTTPResource(@Nonnull @ParameterName(name="backingFile") final String backingFile,
            @Nonnull @ParameterName(name="client") final HttpClient client, 
            @NotEmpty @Nonnull @ParameterName(name="url") final String url) throws IOException {
        super(client, url);
        Constraint.isNotNull(backingFile, "File Name must not be null");
        final File file = new File(backingFile);
        backingResource = new FileSystemResource(file);
    }

    /**
     * Constructor.
     * 
     * @param backingFile the file to use as backing store
     * @param client the client we use to connect with.
     * @param url URL to the remote data
     * @throws IOException if the URL was badly formed
     */
    public FileBackedHTTPResource(@Nonnull @ParameterName(name="backingFile") final String backingFile, 
            @Nonnull @ParameterName(name="client") final HttpClient client,
            @Nonnull  @ParameterName(name="url") final URL url)
            throws IOException {
        super(client, url);
        Constraint.isNotNull(backingFile, "File Name must not be null");
        final File file = new File(backingFile);
        backingResource = new FileSystemResource(file);
    }

    /**
     * saveAndClone. Read the contents into memory and then write out to the backing file. Finally
     * 
     * @param input the input stream
     * @return the cloned stream.
     * @throws IOException if an error happens. If the backing file might have been corrupted we delete it.
     */

    @Nonnull protected InputStream saveAndClone(@Nonnull final InputStream input) throws IOException {
        try (final FileOutputStream out = new FileOutputStream(backingResource.getFile())) {
            log.debug("{}: Copying file.", getDescription());
            ByteStreams.copy(input, out);
            log.debug("{}: Copy done.", getDescription());
        } catch (final IOException e) {
            // try to tidy up
            backingResource.getFile().delete();
            log.error("{}: Copy failed: {}", getDescription(), e.getMessage());
            throw e;
        }

        return new FileInputStream(backingResource.getFile());
    }

    /** {@inheritDoc} */
    @Override @Nonnull public InputStream getInputStream() throws IOException {
        try (final InputStream stream = super.getInputStream()) {
            return saveAndClone(stream);
        } catch (final IOException ex) {
            log.debug("{} Error obtaining HTTPResource InputStream or creating backing file", getDescription(), ex);
            log.warn("{} HTTP resource was inaccessible for getInputStream(), trying backing file.", getDescription());
            try {
                return new FileInputStream(backingResource.getFile());
            } catch (final IOException e) {
                log.error("FileBackedHTTPResource {}: Could not read backing file: {}", getDescription(),
                        e.getMessage());
                throw e;
            }
        }
    }

    /** {@inheritDoc} */
    @Override public boolean exists() {

        log.debug("{}: Attempting to fetch HTTP resource", getDescription());
        final HttpResponse response;
        try {
            response = getResourceHeaders();
        } catch (final IOException e) {
            log.info("{}: Could not reach URL, trying file", getDescription(), e);
            return backingResource.exists();
        }
        final int httpStatusCode = response.getCode();

        if (httpStatusCode == HttpStatus.SC_OK) {
            return true;
        }
        return backingResource.exists();
    }

    /** {@inheritDoc} */
    @Override public long contentLength() throws IOException {

        try {
            return super.contentLength();
        } catch (final IOException e) {
            log.info("{}: Could not reach URL, trying file", getDescription(), e);
            return backingResource.contentLength();
        }
    }

    /** {@inheritDoc} */
    @Override public long lastModified() throws IOException {
        try {
            return super.lastModified();
        } catch (final IOException e) {
            log.info("{}: Could not reach URL, trying file", getDescription(), e);
            return backingResource.lastModified();
        }
    }

    /** {@inheritDoc} */
    @Override @Nonnull public HTTPResource createRelative(@Nonnull final String relativePath) throws IOException {
        log.warn("{}: Relative resources are not file backed");
        return super.createRelative(relativePath);
    }

    /** {@inheritDoc} */
    @Override @Nonnull public String getDescription() {
        String urlAsString;
        try {
            urlAsString = getURL().toString();
        } catch (final IOException e) {
            urlAsString = "<unknown>";
        }

        final StringBuilder builder =
                new StringBuilder("FileBackedHTTPResource [").append(urlAsString).append('|')
                        .append(backingResource.getDescription()).append(']');
        final String result = builder.toString();
        assert result != null;
        return result;
    }
    
}