/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet;

import java.util.Collection;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;

import jakarta.servlet.ServletRequest;
import jakarta.servlet.http.HttpServletRequest;
import net.shibboleth.shared.annotation.ParameterName;
import net.shibboleth.shared.collection.CollectionSupport;

/**
 * Predicate based on comparing a request URL to a set of matching prefixes. 
 */
public class RequestURLPrefixPredicate implements Predicate<ServletRequest> {

    /** Prefixes to check for. */
    @Nonnull private Collection<URLPrefix> matchingPrefixes;

    /**
     * Constructor.
     * 
     * @param prefixes prefixes to match 
     */
    @Autowired
    public RequestURLPrefixPredicate(@Nullable @ParameterName(name="prefixes") final Collection<URLPrefix> prefixes) {
        if (prefixes != null) {
            matchingPrefixes = CollectionSupport.copyToList(prefixes);
        } else {
            matchingPrefixes = CollectionSupport.emptyList();
        }
    }
    
    /** {@inheritDoc} */
    public boolean test(@Nullable final ServletRequest input) {

        if (input instanceof HttpServletRequest req) {
            final String uri = req.getRequestURI();
            for (final URLPrefix p : matchingPrefixes) {
                if (uri.startsWith(input.getServletContext().getContextPath() +  p.getValue())) {
                    return true;
                }
            }
        }
        
        return false;
    }

}