/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.servlet.impl;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;

import jakarta.servlet.Filter;
import jakarta.servlet.FilterChain;
import jakarta.servlet.FilterConfig;
import jakarta.servlet.ServletException;
import jakarta.servlet.ServletRequest;
import jakarta.servlet.ServletResponse;
import net.shibboleth.shared.servlet.AbstractConditionalFilter;
import net.shibboleth.shared.spring.servlet.ChainableFilter;


/**
 * Implementation of {@link Filter} that encapsulates and runs a chain of embedded filters in a defined
 * order.
 * 
 * <p>This is provided to deal with the problem of filter order when programmatic registration is done.
 * Good or bad, some filters are order-sensitive and wishing that wasn't true doesn't change it.</p> 
 */
public class ChainingFilter implements Filter {

    /** Embedded chain. */
    @Nullable private List<ChainableFilter> filters;
    
    /**
     * Constructor.
     *  
     * @param filterChain auto-wired chain of filters to run
     */
    @Autowired
    public ChainingFilter(@Nullable final List<ChainableFilter> filterChain) {
        if (filterChain != null) {
            filters = List.copyOf(filterChain);
        } else {
            filters = null;
        }
    }
    
    /** {@inheritDoc} */
    public void init(final FilterConfig filterConfig) throws ServletException {
    }

    /** {@inheritDoc} */
    public void destroy() {
    }

    /** {@inheritDoc} */
    
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain)
            throws IOException, ServletException {
        
        if (filters != null && !filters.isEmpty()) {
            new Chain(chain).doFilter(request, response);
        } else {
            chain.doFilter(request, response);
        }
    }

    /** Internal iteration of chain. */
    private class Chain implements FilterChain {

        /** Tracked iterator. */
        @Nonnull private Iterator<ChainableFilter> iterator;
        
        /** Chain of top-level filters. */
        @Nonnull private FilterChain outerChain;
        
        /**
         * Constructor.
         * 
         * @param outer outer filter chain
         */
        public Chain(@Nonnull final FilterChain outer) {
            assert filters != null ;
            iterator = filters.iterator();
            outerChain = outer;
        }
        
        /** {@inheritDoc} */
        @Override
        public void doFilter(final ServletRequest request, final ServletResponse response)
                throws IOException, ServletException {
            if (iterator.hasNext()) {
                final ChainableFilter filter = iterator.next();
                if (filter instanceof AbstractConditionalFilter &&
                        !((AbstractConditionalFilter) filter).getActivationCondition().test(request)) {
                    doFilter(request, response);
                    return;
                }
                
                filter.doFilter(request, response, this);
            } else {
                outerChain.doFilter(request, response);
            }
        }
    }

}