/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.httpclient;

import java.io.IOException;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.hc.client5.http.protocol.HttpClientContext;
import org.apache.hc.core5.http.ClassicHttpRequest;

/**
 * Extension that allows the {@link HttpClientContext} to be externally manipulated before use.
 * 
 * @since 7.4.0
 */
@ThreadSafe
public interface HttpClientContextHandler {
    
    /**
     * Perform any desired context modifications before use.
     * 
     * @param context the context to operate on
     * @param request the request that will be executed
     * 
     * @throws IOException if the call should be aborted
     */
    void invokeBefore(@Nonnull final HttpClientContext context, @Nonnull final ClassicHttpRequest request)
        throws IOException;
    
    /**
     * Perform any desired context modifications after use.
     * 
     * @param context the context to operate on
     * @param request the request that was executed
     * 
     * @throws IOException if the call should be aborted
     */
    void invokeAfter(@Nonnull final HttpClientContext context, @Nonnull final ClassicHttpRequest request)
        throws IOException;
    
}