/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.idp.saml.attribute.transcoding;

import java.util.List;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.opensaml.core.xml.XMLObject;
import org.opensaml.core.xml.config.XMLObjectProviderRegistrySupport;
import org.opensaml.profile.context.ProfileRequestContext;
import org.opensaml.saml.common.SAMLObjectBuilder;
import org.opensaml.saml.saml1.core.Attribute;
import org.opensaml.saml.saml1.core.AttributeDesignator;
import org.slf4j.Logger;

import com.google.common.base.Strings;

import net.shibboleth.idp.attribute.AttributeDecodingException;
import net.shibboleth.idp.attribute.AttributeEncodingException;
import net.shibboleth.idp.attribute.IdPAttribute;
import net.shibboleth.idp.attribute.IdPAttributeValue;
import net.shibboleth.idp.attribute.IdPRequestedAttribute;
import net.shibboleth.idp.attribute.transcoding.AttributeTranscoderRegistry;
import net.shibboleth.idp.attribute.transcoding.TranscodingRule;
import net.shibboleth.idp.saml.xml.SAMLConstants;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Base class for transcoders that operate on a SAML 1 {@link Attribute} or {@link AttributeDesignator}.
 * 
 * @param <EncodedType> the type of data that can be handled by the transcoder
 */
public abstract class AbstractSAML1AttributeTranscoder<EncodedType extends IdPAttributeValue>
        extends AbstractSAMLAttributeTranscoder<AttributeDesignator,EncodedType>
        implements SAML1AttributeTranscoder<EncodedType> {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(AbstractSAML1AttributeTranscoder.class);
    
    /** Builder used to construct {@link Attribute} objects. */
    @Nonnull private final SAMLObjectBuilder<Attribute> attributeBuilder;

    /** Builder used to construct {@link AttributeDesignator} objects. */
    @Nonnull private final SAMLObjectBuilder<AttributeDesignator> designatorBuilder;
    
    /** Constructor. */
    public AbstractSAML1AttributeTranscoder() {
        attributeBuilder = (SAMLObjectBuilder<Attribute>)
                XMLObjectProviderRegistrySupport.getBuilderFactory().<Attribute>ensureBuilder(
                        Attribute.TYPE_NAME);
        designatorBuilder = (SAMLObjectBuilder<AttributeDesignator>)
                XMLObjectProviderRegistrySupport.getBuilderFactory().<AttributeDesignator>ensureBuilder(
                        AttributeDesignator.TYPE_NAME);
    }

    /** {@inheritDoc} */
    @Nonnull public Class<AttributeDesignator> getEncodedType() {
        return AttributeDesignator.class;
    }
    
    /** {@inheritDoc} */
    @Nullable public String getEncodedName(@Nonnull final TranscodingRule rule) {
        
        try {
            // SAML 1 naming should be based on only what needs to be available from the properties alone.
            return new NamingFunction().apply(buildAttribute(null, null, AttributeDesignator.class, rule,
                    CollectionSupport.emptyList()));
        } catch (final AttributeEncodingException e) {
            return null;
        }
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected AttributeDesignator buildAttribute(@Nullable final ProfileRequestContext profileRequestContext,
            @Nullable final IdPAttribute attribute, @Nonnull final Class<? extends AttributeDesignator> to,
            @Nonnull final TranscodingRule rule, @Nonnull final List<XMLObject> attributeValues)
                    throws AttributeEncodingException {

        if (attribute != null && !attribute.getValues().isEmpty() && attributeValues.isEmpty()) {
            throw new AttributeEncodingException("Failed to encode any values for attribute " + attribute.getId());
        }

        final AttributeDesignator samlAttribute;
        
        if (to.equals(Attribute.class)) {
            if (attributeValues.isEmpty()) {
                throw new AttributeEncodingException("Unable to encode a SAML 1 Attribute with no values");
            }
            
            samlAttribute = attributeBuilder.buildObject();
            ((Attribute) samlAttribute).getAttributeValues().addAll(attributeValues);
        } else if (to.equals(AttributeDesignator.class)) {
            samlAttribute = designatorBuilder.buildObject();
            if (!attributeValues.isEmpty()) {
                log.warn("Lossy conversion to AttributeDesignator");
            }
        } else {
            throw new AttributeEncodingException("Unsupported target object type: " + to.getName());
        }

        encodeName(profileRequestContext, attribute, samlAttribute, rule);
        
        return samlAttribute;
    }
    
    protected void encodeName(@Nullable final ProfileRequestContext profileRequestContext,
            @Nullable final IdPAttribute attribute, @Nonnull final AttributeDesignator samlAttribute,
            @Nonnull final TranscodingRule rule) throws AttributeEncodingException {
        
        // Use metadata tag to derive name of Attribute?
        final Boolean useMetadata = rule.getOrDefault(PROP_NAME_FROM_METADATA, Boolean.class, false);
        if (useMetadata != null && useMetadata) {
            final String id = attribute != null ? attribute.getId() :
                rule.get(AttributeTranscoderRegistry.PROP_ID, String.class);
            if (id == null) {
                log.warn("Rule specified {} but no attribute ID available", PROP_NAME_FROM_METADATA);
            } else {
                final String tagValue = getNameFromMetadata(profileRequestContext, id);
                if (tagValue != null) {
                    final int lastSpace = tagValue.lastIndexOf(' ');
                    final String name;
                    final String namespace;
                    if (lastSpace > 0) {
                        name = StringSupport.trimOrNull(tagValue.substring(0, lastSpace));
                        namespace = StringSupport.trimOrNull(tagValue.substring(lastSpace));
                        if (name != null && namespace != null) {
                            samlAttribute.setAttributeName(name);
                            samlAttribute.setAttributeNamespace(namespace);
                            log.debug("Encoding IdPAttribute {} via metadata tag as Name {}, Namespace {}", id,
                                    name, namespace);
                            return;
                        }
                    }
                    log.warn("Metadata tag {}, value {}, was not in the expected form", METADATA_TAG_NAME, tagValue);
                }
            }
        }
        
        final String name = rule.get(PROP_NAME, String.class);
        if (Strings.isNullOrEmpty(name)) {
            throw new AttributeEncodingException("Required transcoder property '" + PROP_NAME + "' not found");
        }
        
        samlAttribute.setAttributeName(name);
        samlAttribute.setAttributeNamespace(
                rule.getOrDefault(PROP_NAMESPACE, String.class, SAMLConstants.SAML1_ATTR_NAMESPACE_URI));
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected IdPAttribute buildIdPAttribute(
            @Nullable final ProfileRequestContext profileRequestContext, @Nonnull final AttributeDesignator attribute,
            @Nonnull final TranscodingRule rule, @Nonnull final List<IdPAttributeValue> attributeValues)
                    throws AttributeDecodingException {
        
        final String id = rule.get(AttributeTranscoderRegistry.PROP_ID, String.class);
        if (Strings.isNullOrEmpty(id)) {
            throw new AttributeDecodingException("Required transcoder property 'id' not found");
        }
        assert id != null;
        
        final IdPAttribute idpAttribute;
        if (attribute instanceof Attribute) {
            if (!((Attribute) attribute).getAttributeValues().isEmpty() && attributeValues.isEmpty()) {
                throw new AttributeDecodingException("Failed to decode any values for attribute "
                        + attribute.getAttributeName());
            }
            idpAttribute = new IdPAttribute(id);
        } else {
            idpAttribute = new IdPRequestedAttribute(id);
        }
        
        idpAttribute.setValues(attributeValues);
        
        return idpAttribute;
    }
    
    /** {@inheritDoc} */
    @Override
    @Nonnull protected Iterable<XMLObject> getValues(@Nonnull final AttributeDesignator input) {
        return input instanceof Attribute ? ((Attribute) input).getAttributeValues() : CollectionSupport.emptyList();
    }

    /** A function to produce a "canonical" name for a SAML 1 {@link AttributeDesignator} for transcoding rules. */
    public static class NamingFunction implements Function<AttributeDesignator,String> {

        /** {@inheritDoc} */
        @Nullable public String apply(@Nullable final AttributeDesignator input) {
            
            if (input == null || input.getAttributeName() == null || input.getAttributeNamespace() == null) {
                return null;
            }
            
            final StringBuilder builder = new StringBuilder();
            builder.append("SAML1:{").append(input.getAttributeNamespace()).append('}')
                .append(input.getAttributeName());
            return builder.toString();
        }

    }

}