/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.config;

import javax.annotation.Nonnull;

import org.slf4j.Logger;

import org.springframework.beans.factory.config.BeanPostProcessor;

import net.shibboleth.shared.component.IdentifiableComponent;
import net.shibboleth.shared.primitive.LoggerFactory;

/**
 * Pre-processes {@link IdentifiableComponent} beans by setting the bean ID to the bean name.
 */
public class IdentifiableBeanPostProcessor implements BeanPostProcessor {

    /** Logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(IdentifiableBeanPostProcessor.class);

    /** {@inheritDoc} */
    @Nonnull public Object postProcessBeforeInitialization(@Nonnull final Object bean, @Nonnull final String beanName) {
        if (bean instanceof IdentifiableComponent) {
            final IdentifiableComponent component = (IdentifiableComponent) bean;
            final String id = component.getId();
            if (id == null) {
                component.setId(beanName);
            } else if (log.isDebugEnabled()) {
                if (id.equals(beanName)) {
                    log.trace("The 'id' property is redundant for bean with 'id' attribute '{}'", beanName);
                } else {
                    log.trace("The 'id' property is not the same as the 'id' attribute for bean '{}'!='{}'",
                            component.getId(), beanName);
                }
            }
        }
        return bean;
    }

    /** {@inheritDoc} */
    @Nonnull public Object postProcessAfterInitialization(@Nonnull final Object bean, @Nonnull final String beanName) {
        return bean;
    }

}