/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.custom;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.xml.sax.InputSource;

import net.shibboleth.shared.primitive.LoggerFactory;

import org.slf4j.Logger;
import org.springframework.beans.factory.BeanDefinitionStoreException;
import org.springframework.beans.factory.xml.ResourceEntityResolver;
import org.springframework.core.io.ResourceLoader;

/**
 * Modified copy of Spring's existing {@link ResourceEntityResolver} class that
 * elides the fall-through logic allowing for http(s) resolution of entities.
 */
public class LocalOnlyResourceEntityResolver extends ResourceEntityResolver {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(LocalOnlyResourceEntityResolver.class);

    /**
     * Create a ResourceEntityResolver for the specified ResourceLoader
     * (usually, an ApplicationContext).
     * 
     * @param loader the ResourceLoader (or ApplicationContext)
     * to load XML entity includes with
     */
    public LocalOnlyResourceEntityResolver(@Nonnull final ResourceLoader loader) {
        super(loader);
    }
    
    /** {@inheritDoc} */
    @Override
    @Nullable protected InputSource resolveSchemaEntity(@Nullable final String publicId,
            @Nonnull final String systemId) {
        
        log.warn("Blocking attempted remote resolution of [{}]", systemId);
        // If we don't throw here, Java's broken parser just blindly proceeds with its own
        // internal entity resolution.
        throw new BeanDefinitionStoreException("Blocked atttempted remote resolution");
    }

}