/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.custom;

import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.xml.DelegatingEntityResolver;
import org.springframework.beans.factory.xml.XmlBeanDefinitionReader;
import org.springframework.core.io.ResourceLoader;

/**
 * An extension to the standard {@link XmlBeanDefinitionReader} that defaults some settings.
 * <ul>
 * <li>documentReaderClass property is set to {@link SchemaTypeAwareBeanDefinitionDocumentReader}</li>
 * <li>namespaceAware property is set to true</li>
 * <li>validating property is set to false</li>
 * </ul>
 */
public class SchemaTypeAwareXMLBeanDefinitionReader extends XmlBeanDefinitionReader {

    /**
     * Constructor.
     * 
     * @param beanRegistry the bean definition registry that will be populated by this reader
     */
    public SchemaTypeAwareXMLBeanDefinitionReader(final BeanDefinitionRegistry beanRegistry) {
        super(beanRegistry);

        setDocumentReaderClass(SchemaTypeAwareBeanDefinitionDocumentReader.class);

        setValidationMode(VALIDATION_XSD);

        // This installs the appropriate XML EntityResolver with our version if needed.
        final ResourceLoader resourceLoader = getResourceLoader();
        if (resourceLoader != null) {
            setEntityResolver(new LocalOnlyResourceEntityResolver(resourceLoader));
        } else {
            setEntityResolver(new DelegatingEntityResolver(getBeanClassLoader()));
        }
    }

}