/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.factory;

import java.util.regex.Pattern;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.springframework.beans.factory.BeanCreationException;

import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Factory bean for {@link Pattern}. Allows us to inject property based case sensitivity.
 */
public class PatternFactoryBean extends AbstractComponentAwareFactoryBean<Pattern> {

    /** Whether the we are case sensitive or not. */
    @Nullable private String caseSensitive;

    /** The regular expressions. */
    @Nullable private String regexp;

    /** {@inheritDoc} */
    @Override
    @Nonnull public Class<?> getObjectType() {
        return Pattern.class;
    }

    /**
     * Get case sensitivity.
     * 
     * @return the case sensitivity
     */
    @Nullable public String getCaseSensitive() {
        return caseSensitive;
    }

    /**
     * Set case sensitivity.
     * 
     * @param what value to set
     */
    public void setCaseSensitive(@Nullable final String what) {
        caseSensitive = what;
    }

    /**
     * Get the regular expression.
     * 
     * @return the regexp
     */
    @Nullable public String getRegexp() {
        return regexp;
    }

    /**
     * Set the regular expression.
     * 
     * @param what what to set
     */
    public void setRegexp(@Nullable final String what) {
        regexp = what;
    }

    /** {@inheritDoc} */
    @Override
    @Nonnull protected Pattern doCreateInstance() throws Exception {
        if (regexp == null) {
            throw new BeanCreationException("Regular expression cannot be null");
        }
        
        final Boolean isCaseSensitive;
        if (caseSensitive != null) {
            isCaseSensitive = StringSupport.booleanOf(caseSensitive);
            Constraint.isNotNull(isCaseSensitive, "caseSensitive cannot be null");
        } else {
            isCaseSensitive = true;
        }

        return Pattern.compile(regexp, isCaseSensitive ? 0 : Pattern.CASE_INSENSITIVE);
    }

}