/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.spring.util;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.collection.CollectionSupport;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.LoggerFactory;
import net.shibboleth.shared.primitive.StringSupport;
import net.shibboleth.shared.spring.config.BooleanToPredicateConverter;
import net.shibboleth.shared.spring.config.StringBooleanToPredicateConverter;
import net.shibboleth.shared.spring.config.StringToDurationConverter;
import net.shibboleth.shared.spring.config.StringToIPRangeConverter;
import net.shibboleth.shared.spring.config.StringToResourceConverter;
import net.shibboleth.shared.spring.context.FilesystemGenericApplicationContext;
import net.shibboleth.shared.spring.custom.SchemaTypeAwareXMLBeanDefinitionReader;
import net.shibboleth.shared.spring.resource.ConditionalResourceResolver;
import net.shibboleth.shared.spring.resource.PreferFileSystemResourceLoader;

import org.slf4j.Logger;

import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationContextInitializer;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.support.ConversionServiceFactoryBean;
import org.springframework.context.support.GenericApplicationContext;
import org.springframework.context.support.PropertySourcesPlaceholderConfigurer;
import org.springframework.core.convert.ConversionService;
import org.springframework.core.convert.converter.Converter;
import org.springframework.core.env.PropertySource;
import org.springframework.core.io.Resource;

/**
 * Fluent builder for a {@link FilesystemGenericApplicationContext} equipped with various standard features,
 * behavior, converters, etc. that are applicable to the Shibboleth software components.
 * 
 * @since 5.4.0
 */
public class ApplicationContextBuilder {

    /** Class logger. */
    @Nonnull private final Logger log = LoggerFactory.getLogger(ApplicationContextBuilder.class);

    /** Context name. */
    @Nullable @NotEmpty private String contextName;
    
    /** Unresolved configuration sources for this service. */
    @Nullable private List<String> configurationSources;

    /** Configuration resources for this service. */
    @Nullable private List<Resource> configurationResources;

    /** Conversion service to use. */
    @Nullable private ConversionService conversionService;
    
    /** List of context initializers. */
    @Nullable private List<ApplicationContextInitializer<? super FilesystemGenericApplicationContext>>
    contextInitializers;

    /** List of bean factory post processors for this service's content. */
    @Nullable private List<BeanFactoryPostProcessor> factoryPostProcessors;

    /** List of bean post processors for this service's content. */
    @Nullable private List<BeanPostProcessor> postProcessors;
    
    /** List of property sources to add. */
    @Nullable private List<PropertySource<?>> propertySources;
    
    /** Bean profiles to enable. */
    @Nullable private Collection<String> beanProfiles;

    /** Application context owning this engine. */
    @Nullable private ApplicationContext parentContext;
    
    /** Whether to install a JVM shutdown hook. */
    private boolean installShutdownHook;
    
    /**
     * Set the name of the context.
     * 
     * @param name name
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setName(@Nullable @NotEmpty final String name) {
        contextName = StringSupport.trimOrNull(name);
        
        return this;
    }

    /**
     * Set a conversion service to use.
     * 
     * @param service conversion service
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setConversionService(@Nullable final ConversionService service) {
        conversionService = service;
        
        return this;
    }
    
    /**
     * Set a single configuration resource for this context.
     * 
     * @param config configuration for this context
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setServiceConfiguration(@Nonnull final Resource config) {
        configurationResources = CollectionSupport.singletonList(
                Constraint.isNotNull(config, "Resource cannot be null"));
        
        return this;
    }

    /**
     * Set the unresolved configurations for this context.
     * 
     * <p>This method is used to allow the context to resolve the resources.</p>
     * 
     * @param configs unresolved configurations for this context
     * 
     * @return this builder
     * 
     * @since 7.0.0
     */
    @Nonnull public ApplicationContextBuilder setUnresolvedServiceConfigurations(
            @Nonnull final Collection<String> configs) {
        configurationSources = CollectionSupport.copyToList(
                Constraint.isNotNull(configs, "Service configurations cannot be null"));
        
        return this;
    }

    /**
     * Set the configurations for this context.
     * 
     * @param configs configurations for this context
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setServiceConfigurations(
            @Nonnull final Collection<Resource> configs) {
        configurationResources = CollectionSupport.copyToList(
                Constraint.isNotNull(configs, "Service configurations cannot be null"));
        
        return this;
    }
    
    /**
     * Set additional property sources for this context.
     * 
     * @param sources property sources to add
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setPropertySources(
            @Nonnull final List<PropertySource<?>> sources) {
        propertySources = List.copyOf(Constraint.isNotNull(sources, "Property sources cannot be null"));
        
        return this;
    }
    

    /**
     * Set a single context initializer for this context.
     * 
     * @param initializer initializer to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setContextInitializer(
            @Nonnull final ApplicationContextInitializer<? super FilesystemGenericApplicationContext> initializer) {
        Constraint.isNotNull(initializer, "ApplicationContextInitializer cannot be null");
        
        contextInitializers = CollectionSupport.singletonList(initializer);
        
        return this;
    }
    
    /**
     * Set the list of context initializers for this context.
     * 
     * @param initializers initializers to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setContextInitializers(
            @Nonnull
            final List<ApplicationContextInitializer<? super FilesystemGenericApplicationContext>> initializers) {
        contextInitializers = List.copyOf(Constraint.isNotNull(initializers, "Context initializers cannot be null"));
        
        return this;
    }
    
    /**
     * Set a single bean factory post processor for this context.
     * 
     * <p>Note that if this object is {@link ApplicationContextAware} or {@link EnvironmentAware},
     * that is injected by the builder. Do NOT pass in processors that already contain references to
     * a parent context.</p>
     * 
     * @param processor bean factory post processor to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setBeanFactoryPostProcessor(
            @Nonnull final BeanFactoryPostProcessor processor) {
        Constraint.isNotNull(processor, "BeanFactoryPostProcessor cannot be null");
        
        factoryPostProcessors = CollectionSupport.singletonList(processor);
        
        return this;
    }


    /**
     * Set the list of bean factory post processors for this context.
     * 
     * <p>Note that if these objects are {@link ApplicationContextAware} or {@link EnvironmentAware},
     * that is injected by the builder. Do NOT pass in processors that already contain references to
     * a parent context.</p>
     * 
     * @param processors bean factory post processors to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setBeanFactoryPostProcessors(
            @Nonnull final List<BeanFactoryPostProcessor> processors) {        
        Constraint.isNotNull(processors, "BeanFactoryPostProcessor collection cannot be null");

        factoryPostProcessors = CollectionSupport.copyToList(processors);
        
        return this;
    }
    
    /**
     * Set a single bean post processor for this context.
     * 
     * <p>Note that if this object is {@link ApplicationContextAware} or {@link EnvironmentAware},
     * that is injected by the builder. Do NOT pass in processors that already contain references to
     * a parent context.</p>
     * 
     * @param processor bean post processor to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setBeanPostProcessor(@Nonnull final BeanPostProcessor processor) {
        Constraint.isNotNull(processor, "BeanPostProcessor cannot be null");
        
        postProcessors = CollectionSupport.singletonList(processor);
        
        return this;
    }

    /**
     * Set the list of bean post processors for this context.
     * 
     * <p>Note that if these objects are {@link ApplicationContextAware} or {@link EnvironmentAware},
     * that is injected by the builder. Do NOT pass in processors that already contain references to
     * a parent context.</p>
     * 
     * @param processors bean post processors to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setBeanPostProcessors(
            @Nonnull final List<BeanPostProcessor> processors) {        
        Constraint.isNotNull(processors, "BeanPostProcessor collection cannot be null");

        postProcessors = CollectionSupport.copyToList(processors);
        
        return this;
    }
    
    /**
     * Set the bean profiles for this context.
     * 
     * @param profiles bean profiles to apply
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setBeanProfiles(@Nonnull final Collection<String> profiles) {
        beanProfiles = StringSupport.normalizeStringCollection(profiles);
        
        return this;
    }
    
    /**
     * Set the parent context.
     * 
     * @param context parent context
     * 
     * @return this builder
     */
    @Nonnull public ApplicationContextBuilder setParentContext(@Nullable final ApplicationContext context) {
        parentContext = context;
        
        return this;
    }
    
    /**
     * Set whether to install a JVM shutdown hook.
     * 
     * <p>Defaults to false.</p>
     * 
     * @param flag flag to set
     * 
     * @return this builder
     * 
     * @since 7.0.0
     */
    @Nonnull public ApplicationContextBuilder installShutdownHook(final boolean flag) {
        installShutdownHook = flag;
        
        return this;
    }
    
// Checkstyle: CyclomaticComplexity|MethodLength OFF
    /**
     * Build a {@link GenericApplicationContext} context.
     * 
     * @return the built context, initialized and loaded
     */
    @Nonnull public GenericApplicationContext build() {
        
        final FilesystemGenericApplicationContext context = new FilesystemGenericApplicationContext(parentContext);
        context.setDisplayName("ApplicationContext:" + (contextName != null ? contextName : "anonymous"));
        
        final PreferFileSystemResourceLoader loader = new PreferFileSystemResourceLoader();
        loader.addProtocolResolver(new ConditionalResourceResolver());
        context.setResourceLoader(loader);
        
        // With the post-processors and converters, we need to make sure to check for a need to
        // inject the ApplicationContext or Environment. Spring won't do this since these objects
        // were created outside and not inside the context.
        
        if (conversionService != null) {
            context.getBeanFactory().setConversionService(conversionService);
        } else {
            final ConversionServiceFactoryBean service = new ConversionServiceFactoryBean();
            final Set<Converter<?,?>> converters = CollectionSupport.setOf(
                    new StringToIPRangeConverter(),
                    new BooleanToPredicateConverter(),
                    new StringBooleanToPredicateConverter(),
                    new StringToResourceConverter(),
                    new StringToDurationConverter());
            for (final Converter<?,?> c : converters) {
                if (c instanceof ApplicationContextAware aware) {
                    aware.setApplicationContext(context);
                }
                if (c instanceof EnvironmentAware aware) {
                    aware.setEnvironment(context.getEnvironment());
                }
            }
            service.setConverters(converters);
            service.afterPropertiesSet();
            context.getBeanFactory().setConversionService(service.getObject());
        }
        
        boolean needPropertyConfigurer = true;
        
        if (factoryPostProcessors != null) {
            for (final BeanFactoryPostProcessor bfpp : factoryPostProcessors) {
                assert bfpp != null;
                context.addBeanFactoryPostProcessor(bfpp);
                if (bfpp instanceof PropertySourcesPlaceholderConfigurer) {
                    needPropertyConfigurer = false;
                }
                if (bfpp instanceof ApplicationContextAware aware) {
                    aware.setApplicationContext(context);
                }
                if (bfpp instanceof EnvironmentAware aware) {
                    aware.setEnvironment(context.getEnvironment());
                }
            }
        }

        // Auto-install property replacement if needed.
        if (needPropertyConfigurer) {
            final PropertySourcesPlaceholderConfigurer propertyConfigurer =
                    new PropertySourcesPlaceholderConfigurer();
            propertyConfigurer.setPlaceholderPrefix("%{");
            propertyConfigurer.setPlaceholderSuffix("}");
            propertyConfigurer.setEnvironment(context.getEnvironment());
            context.getBeanFactoryPostProcessors().add(propertyConfigurer);
        }
        
        if (postProcessors != null) {
            for (final BeanPostProcessor bpp : postProcessors) {
                assert bpp != null;
                context.getBeanFactory().addBeanPostProcessor(bpp);
                if (bpp instanceof ApplicationContextAware aware) {
                    aware.setApplicationContext(context);
                }
                if (bpp instanceof EnvironmentAware aware) {
                    aware.setEnvironment(context.getEnvironment());
                }
            }
        }
        
        if (beanProfiles != null) {
            final String[] profiles = beanProfiles.toArray(new String[0]);
            assert profiles != null;
            context.getEnvironment().setActiveProfiles(profiles);
        }
        
        if (propertySources != null) {
            propertySources.forEach(p -> context.getEnvironment().getPropertySources().addLast(p));
            context.getEnvironment().setPlaceholderPrefix("%{");
            context.getEnvironment().setPlaceholderSuffix("}");
        }
        
        if (installShutdownHook) {
            context.registerShutdownHook();
        }

        final SchemaTypeAwareXMLBeanDefinitionReader beanDefinitionReader =
                new SchemaTypeAwareXMLBeanDefinitionReader(context);
        
        if (configurationSources != null) {
            configurationSources.stream().forEachOrdered(
                    s -> {
                        try {
                            assert s != null;
                            final Resource[] loaded = context.getResources(s);
                            if (loaded != null && loaded.length > 0) {
                                log.debug("Resolved resources: {}", Arrays.asList(loaded));
                                beanDefinitionReader.loadBeanDefinitions(loaded);
                            } else {
                                log.debug("No resources resolved from {}", s);
                            }
                        } catch (final IOException e) {
                            log.warn("Error loading beans from {}", s, e);
                        }
                    }
                );
        }

        if (configurationResources != null) {
            @Nonnull final List<Resource> filtered = configurationResources.stream()
                .filter(r -> {
                    if (r.exists()) {
                        return true;
                    }
                    log.info("Skipping non-existent resource: {}", r);
                    return false;
                })
                .collect(CollectionSupport.nonnullCollector(Collectors.toUnmodifiableList()))
                .get();
            if (!filtered.isEmpty()) {
                beanDefinitionReader.loadBeanDefinitions(filtered.toArray(new Resource[0]));
            }
        }
        
        if (contextInitializers != null) {
            contextInitializers.forEach(i -> i.initialize(context));
        }

        context.refresh();
        return context;
    }
// Checkstyle: CyclomaticComplexity|MethodLength ON    
}