/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.testing;

import javax.annotation.Nonnull;

import net.shibboleth.shared.annotation.constraint.NotEmpty;
import net.shibboleth.shared.logic.Constraint;
import net.shibboleth.shared.primitive.StringSupport;

/**
 * Support class for working with the project version control repository.
 * 
 * <p>NOTE: If this changes, change the "additional" copy in shib-spring.</p>
 */
public final class RepositorySupport {
    
    /** Constructor. */
    private RepositorySupport() { }

    
    /**
     * Build an HTTPS resource URL for the selected repository name and path on the main branch.
     * 
     * @param repoName the repository name.  If Git, do not include the ".git" suffix.
     * @param resourcePath The relative resource path within the repository, e.g. "foo/bar/baz/file.txt"
     * 
     * @return the HTTPS resource URL
     */
    @Nonnull public static String buildHTTPSResourceURL(@Nonnull final String repoName,
            @Nonnull final String resourcePath) {
        return buildHTTPResourceURL(repoName, resourcePath, true, "HEAD");
    }
    
    /**
     * Build an HTTP/HTTPS resource URL for the selected repository name and path on the main branch.
     * 
     * @param repoName the repository name.  If Git, do not include a trailing ".git" suffix for bare repos.
     * @param resourcePath The relative resource path within the repository, e.g. "foo/bar/baz/file.txt"
     * @param https if true, use https if possible, otherwise use http
     * 
     * @return the HTTP(S) resource URL
     */
    @Nonnull public static String buildHTTPResourceURL(@Nonnull final String repoName,
            @Nonnull final String resourcePath, final boolean https) {
        return buildHTTPResourceURL(repoName, resourcePath, https, "HEAD");
    }
    
    /**
     * Build an HTTP/HTTPS resource URL for the selected repository name and path.
     * 
     * @param repoName the repository name.  If Git, do not include a trailing ".git" suffix for bare repos.
     * @param resourcePath The relative resource path within the repository, e.g. "foo/bar/baz/file.txt"
     * @param https if true, use https if possible, otherwise use http
     * @param branch code branch
     * 
     * @return the HTTP(S) resource URL
     * 
     * @since 8.2.0
     */
    @Nonnull public static String buildHTTPResourceURL(@Nonnull final String repoName,
            @Nonnull final String resourcePath, final boolean https, @Nonnull @NotEmpty final String branch) {
        
        final String repo = Constraint.isNotNull(StringSupport.trimOrNull(repoName), 
                "Repository name was null or empty");
        String path = Constraint.isNotNull(StringSupport.trimOrNull(resourcePath), 
                "Resource path was null or empty");
        
        if (path.startsWith("/")) {
            path = path.substring(1);
        }
        
        if (https) {
            return String.format("https://test.shibboleth.net/unittests/%s/%s/%s", repo, branch, path);
        }
        return String.format("http://git.shibboleth.net/unittests/%s/%s/%s", repo, branch, path);
    }

}