/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.shibboleth.shared.testing;

import java.io.IOException;
import java.io.InputStream;

import javax.annotation.Nonnull;

import org.springframework.core.io.Resource;

/**
 * Compare two resources for equality.
 * 
 * <p>NOTE: If this changes, change the "additional" copy in shib-spring.</p>
 */
public final class ResourceTestHelper {

    /** Hidden constructor. */
    private ResourceTestHelper() {}
    
    /** Compare to see whether the content of two resources are the same.
     * 
     * @param first Resource to compare
     * @param second  Resource to compare
     * 
     * @return whether They are equal
     * 
     * @throws IOException if a read fails
     */
    public static boolean compare(@Nonnull final Resource first, @Nonnull final Resource second) throws IOException {
        try (final InputStream firstStream = first.getInputStream();
                final InputStream secondStream = second.getInputStream()) {
            while (true) {

                // Remove any differences based on CRLF handling
                int firstInt = firstStream.read();
                while (firstInt == 10 || firstInt == 13) {
                    firstInt = firstStream.read();
                }
                int secondInt = secondStream.read();
                while (secondInt == 10 || secondInt == 13) {
                    secondInt = secondStream.read();
                }
                if (firstInt == -1) {
                    return secondInt == -1;
                }
                if (firstInt != secondInt) {
                    return false;
                }
            }
        }
    }

}