/*
 * Decompiled with CFR 0.152.
 */
package org.jbpm.process.core.timer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;
import java.util.regex.Matcher;
import org.jbpm.process.core.timer.BusinessCalendar;
import org.jbpm.process.core.timer.DateTimeUtils;
import org.jbpm.util.PatternConstants;
import org.kie.api.time.SessionClock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BusinessCalendarImpl
implements BusinessCalendar {
    private static final Logger logger = LoggerFactory.getLogger(BusinessCalendarImpl.class);
    private Properties businessCalendarConfiguration;
    private static final long HOUR_IN_MILLIS = 3600000L;
    private int daysPerWeek;
    private int hoursInDay;
    private int startHour;
    private int endHour;
    private String timezone;
    private List<TimePeriod> holidays;
    private List<Integer> weekendDays = new ArrayList<Integer>();
    private SessionClock clock;
    private static final int SIM_WEEK = 3;
    private static final int SIM_DAY = 5;
    private static final int SIM_HOU = 7;
    private static final int SIM_MIN = 9;
    private static final int SIM_SEC = 11;
    public static final String DAYS_PER_WEEK = "business.days.per.week";
    public static final String HOURS_PER_DAY = "business.hours.per.day";
    public static final String START_HOUR = "business.start.hour";
    public static final String END_HOUR = "business.end.hour";
    public static final String HOLIDAYS = "business.holidays";
    public static final String HOLIDAY_DATE_FORMAT = "business.holiday.date.format";
    public static final String WEEKEND_DAYS = "business.weekend.days";
    public static final String TIMEZONE = "business.cal.timezone";
    private static final String DEFAULT_PROPERTIES_NAME = "/jbpm.business.calendar.properties";

    public BusinessCalendarImpl() {
        this(System.getProperty("jbpm.business.calendar.properties", DEFAULT_PROPERTIES_NAME));
    }

    public BusinessCalendarImpl(String propertiesLocation) {
        this.businessCalendarConfiguration = new Properties();
        try (InputStream in = this.getInputStreamProperties(propertiesLocation);){
            this.businessCalendarConfiguration.load(in);
        }
        catch (IOException e) {
            logger.error("Error while loading properties for business calendar {}", (Object)propertiesLocation, (Object)e);
        }
        this.init();
    }

    public BusinessCalendarImpl(Properties configuration) {
        this.businessCalendarConfiguration = configuration;
        this.init();
    }

    public BusinessCalendarImpl(Properties configuration, SessionClock clock) {
        this.businessCalendarConfiguration = configuration;
        this.clock = clock;
        this.init();
    }

    private InputStream getInputStreamProperties(String propertiesLocation) throws IOException {
        InputStream in = this.getClass().getResourceAsStream(propertiesLocation);
        if (in != null) {
            return in;
        }
        File file = new File(propertiesLocation);
        if (file.exists()) {
            return new FileInputStream(file);
        }
        logger.warn("BusinessCalendarImpl was not able to find properties in {}", (Object)propertiesLocation);
        return new InputStream(){

            @Override
            public int read() {
                return -1;
            }
        };
    }

    protected void init() {
        if (this.businessCalendarConfiguration == null) {
            throw new IllegalArgumentException("BusinessCalendar configuration was not provided.");
        }
        this.daysPerWeek = this.getPropertyAsInt(DAYS_PER_WEEK, "5");
        this.hoursInDay = this.getPropertyAsInt(HOURS_PER_DAY, "8");
        this.startHour = this.getPropertyAsInt(START_HOUR, "9");
        this.endHour = this.getPropertyAsInt(END_HOUR, "17");
        this.holidays = this.parseHolidays();
        this.parseWeekendDays();
        this.timezone = this.businessCalendarConfiguration.getProperty(TIMEZONE);
    }

    protected String adoptISOFormat(String timeExpression) {
        try {
            Duration p = null;
            if (DateTimeUtils.isPeriod(timeExpression)) {
                p = Duration.parse(timeExpression);
            } else if (DateTimeUtils.isNumeric(timeExpression)) {
                p = Duration.of(Long.valueOf(timeExpression), ChronoUnit.MILLIS);
            } else {
                OffsetDateTime dateTime = OffsetDateTime.parse(timeExpression, DateTimeFormatter.ISO_DATE_TIME);
                p = Duration.between(OffsetDateTime.now(), dateTime);
            }
            long days = p.toDays();
            long hours = p.toHours() % 24L;
            long minutes = p.toMinutes() % 60L;
            long seconds = p.getSeconds() % 60L;
            long milis = p.toMillis() % 1000L;
            StringBuffer time = new StringBuffer();
            if (days > 0L) {
                time.append(days + "d");
            }
            if (hours > 0L) {
                time.append(hours + "h");
            }
            if (minutes > 0L) {
                time.append(minutes + "m");
            }
            if (seconds > 0L) {
                time.append(seconds + "s");
            }
            if (milis > 0L) {
                time.append(milis + "ms");
            }
            return time.toString();
        }
        catch (Exception e) {
            return timeExpression;
        }
    }

    @Override
    public long calculateBusinessTimeAsDuration(String timeExpression) {
        timeExpression = this.adoptISOFormat(timeExpression);
        Date calculatedDate = this.calculateBusinessTimeAsDate(timeExpression);
        return calculatedDate.getTime() - this.getCurrentTime();
    }

    @Override
    public Date calculateBusinessTimeAsDate(String timeExpression) {
        Matcher mat;
        timeExpression = this.adoptISOFormat(timeExpression);
        String trimmed = timeExpression.trim();
        int weeks = 0;
        int days = 0;
        int hours = 0;
        int min = 0;
        int sec = 0;
        if (trimmed.length() > 0 && (mat = PatternConstants.SIMPLE_TIME_DATE_MATCHER.matcher(trimmed)).matches()) {
            weeks = mat.group(3) != null ? Integer.parseInt(mat.group(3)) : 0;
            days = mat.group(5) != null ? Integer.parseInt(mat.group(5)) : 0;
            hours = mat.group(7) != null ? Integer.parseInt(mat.group(7)) : 0;
            min = mat.group(9) != null ? Integer.parseInt(mat.group(9)) : 0;
            sec = mat.group(11) != null ? Integer.parseInt(mat.group(11)) : 0;
        }
        int remainingSeconds = sec + min * 60 + hours * 3600 + days * this.hoursInDay * 3600 + weeks * this.daysPerWeek * this.hoursInDay * 3600;
        GregorianCalendar c = new GregorianCalendar();
        if (this.timezone != null) {
            ((Calendar)c).setTimeZone(TimeZone.getTimeZone(this.timezone));
        }
        if (this.clock != null) {
            c.setTimeInMillis(this.clock.getCurrentTime());
        }
        while (remainingSeconds > 0 || remainingSeconds == 0 && (!this.isWorkingDay(c) || this.isHoliday(c))) {
            int remainingSecondsToday;
            if (!this.isWorkingDay(c) || this.isHoliday(c)) {
                this.nextDay(c);
                continue;
            }
            int currentHour = c.get(11);
            if (currentHour >= this.endHour) {
                this.nextDay(c);
                continue;
            }
            if (currentHour < this.startHour) {
                this.resetDay(c);
            }
            if (remainingSeconds >= (remainingSecondsToday = (this.endHour - c.get(11)) * 3600 - (c.get(12) * 60 + c.get(13)))) {
                remainingSeconds -= remainingSecondsToday;
                this.nextDay(c);
                continue;
            }
            ((Calendar)c).add(13, remainingSeconds);
            remainingSeconds = 0;
        }
        return c.getTime();
    }

    private void nextDay(Calendar c) {
        c.add(6, 1);
        this.resetDay(c);
    }

    private void resetDay(Calendar c) {
        c.set(11, this.startHour);
        c.set(12, 0);
        c.set(13, 0);
        c.set(14, 0);
    }

    protected void handleHoliday(Calendar c, boolean resetTime) {
        if (!this.holidays.isEmpty()) {
            Date current = c.getTime();
            for (TimePeriod holiday : this.holidays) {
                if (!current.after(holiday.getFrom()) || !current.before(holiday.getTo())) continue;
                GregorianCalendar tmp = new GregorianCalendar();
                tmp.setTime(holiday.getTo());
                GregorianCalendar tmp2 = new GregorianCalendar();
                tmp2.setTime(current);
                tmp2.set(11, 0);
                tmp2.set(12, 0);
                tmp2.set(13, 0);
                tmp2.set(14, 0);
                long difference = tmp.getTimeInMillis() - tmp2.getTimeInMillis();
                c.add(11, (int)(difference / 3600000L));
                this.handleWeekend(c, resetTime);
                break;
            }
        }
    }

    protected int getPropertyAsInt(String propertyName, String defaultValue) {
        String value = this.businessCalendarConfiguration.getProperty(propertyName, defaultValue);
        return Integer.parseInt(value);
    }

    protected List<TimePeriod> parseHolidays() {
        String holidaysString = this.businessCalendarConfiguration.getProperty(HOLIDAYS);
        ArrayList<TimePeriod> holidays = new ArrayList<TimePeriod>();
        int currentYear = Calendar.getInstance().get(1);
        if (holidaysString != null) {
            String[] hPeriods = holidaysString.split(",");
            SimpleDateFormat sdf = new SimpleDateFormat(this.businessCalendarConfiguration.getProperty(HOLIDAY_DATE_FORMAT, "yyyy-MM-dd"));
            for (String hPeriod : hPeriods) {
                boolean addNextYearHolidays = false;
                String[] fromTo = hPeriod.split(":");
                if (fromTo[0].startsWith("*")) {
                    addNextYearHolidays = true;
                    fromTo[0] = fromTo[0].replaceFirst("\\*", currentYear + "");
                }
                try {
                    Date from;
                    if (fromTo.length == 2) {
                        GregorianCalendar tmpFrom = new GregorianCalendar();
                        if (this.timezone != null) {
                            ((Calendar)tmpFrom).setTimeZone(TimeZone.getTimeZone(this.timezone));
                        }
                        tmpFrom.setTime(sdf.parse(fromTo[0]));
                        if (fromTo[1].startsWith("*")) {
                            fromTo[1] = fromTo[1].replaceFirst("\\*", currentYear + "");
                        }
                        GregorianCalendar tmpTo = new GregorianCalendar();
                        if (this.timezone != null) {
                            ((Calendar)tmpTo).setTimeZone(TimeZone.getTimeZone(this.timezone));
                        }
                        tmpTo.setTime(sdf.parse(fromTo[1]));
                        from = tmpFrom.getTime();
                        ((Calendar)tmpTo).add(6, 1);
                        if (tmpFrom.get(2) > tmpTo.get(2) && tmpFrom.get(1) == tmpTo.get(1)) {
                            ((Calendar)tmpTo).add(1, 1);
                        }
                        Date to = tmpTo.getTime();
                        holidays.add(new TimePeriod(from, to));
                        holidays.add(new TimePeriod(from, to));
                        if (!addNextYearHolidays) continue;
                        tmpFrom = new GregorianCalendar();
                        if (this.timezone != null) {
                            ((Calendar)tmpFrom).setTimeZone(TimeZone.getTimeZone(this.timezone));
                        }
                        tmpFrom.setTime(sdf.parse(fromTo[0]));
                        ((Calendar)tmpFrom).add(1, 1);
                        from = tmpFrom.getTime();
                        tmpTo = new GregorianCalendar();
                        if (this.timezone != null) {
                            ((Calendar)tmpTo).setTimeZone(TimeZone.getTimeZone(this.timezone));
                        }
                        tmpTo.setTime(sdf.parse(fromTo[1]));
                        ((Calendar)tmpTo).add(1, 1);
                        ((Calendar)tmpTo).add(6, 1);
                        if (tmpFrom.get(2) > tmpTo.get(2) && tmpFrom.get(1) == tmpTo.get(1)) {
                            ((Calendar)tmpTo).add(1, 1);
                        }
                        to = tmpTo.getTime();
                        holidays.add(new TimePeriod(from, to));
                        continue;
                    }
                    GregorianCalendar c = new GregorianCalendar();
                    c.setTime(sdf.parse(fromTo[0]));
                    ((Calendar)c).add(6, 1);
                    holidays.add(new TimePeriod(sdf.parse(fromTo[0]), c.getTime()));
                    if (!addNextYearHolidays) continue;
                    Calendar tmp = Calendar.getInstance();
                    tmp.setTime(sdf.parse(fromTo[0]));
                    tmp.add(1, 1);
                    from = tmp.getTime();
                    ((Calendar)c).add(1, 1);
                    holidays.add(new TimePeriod(from, c.getTime()));
                }
                catch (Exception e) {
                    logger.error("Error while parsing holiday in business calendar", (Throwable)e);
                }
            }
        }
        return holidays;
    }

    protected void parseWeekendDays() {
        String weekendDays = this.businessCalendarConfiguration.getProperty(WEEKEND_DAYS);
        if (weekendDays == null) {
            this.weekendDays.add(7);
            this.weekendDays.add(1);
        } else {
            String[] days;
            for (String day : days = weekendDays.split(",")) {
                this.weekendDays.add(Integer.parseInt(day));
            }
        }
    }

    protected long getCurrentTime() {
        if (this.clock != null) {
            return this.clock.getCurrentTime();
        }
        return System.currentTimeMillis();
    }

    protected boolean isHoliday(Calendar c) {
        if (this.holidays.isEmpty()) {
            return false;
        }
        Date current = c.getTime();
        for (TimePeriod holiday : this.holidays) {
            if (!current.after(holiday.getFrom()) || !current.before(holiday.getTo())) continue;
            return true;
        }
        return false;
    }

    protected boolean isWorkingDay(Calendar c) {
        return this.isWorkingDay(c.get(7));
    }

    protected boolean isWorkingDay(int day) {
        return !this.weekendDays.contains(day);
    }

    protected void handleWeekend(Calendar c, boolean resetTime) {
        int dayOfTheWeek = c.get(7);
        while (!this.isWorkingDay(dayOfTheWeek)) {
            c.add(6, 1);
            if (resetTime) {
                c.set(11, 0);
                c.set(12, 0);
                c.set(13, 0);
                c.set(14, 0);
            }
            dayOfTheWeek = c.get(7);
        }
    }

    private class TimePeriod {
        private Date from;
        private Date to;

        protected TimePeriod(Date from, Date to) {
            this.from = from;
            this.to = to;
        }

        protected Date getFrom() {
            return this.from;
        }

        protected Date getTo() {
            return this.to;
        }
    }
}

