/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.security.x509;

import java.io.InputStream;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.security.auth.x500.X500Principal;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.opensaml.security.SecurityException;
import org.opensaml.security.crypto.KeySupport;
import org.opensaml.security.crypto.KeySupportTest;
import org.opensaml.security.x509.X509Support;
import org.testng.Assert;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

public class X509SupportTest {
    private PrivateKey entityPrivateKey;
    private String entityPrivKeyBase64 = "MIICXQIBAAKBgQCewxYtt+tEFlhy7+5V3kdCGDATiiXP/A0r2Q+amujuDz0aauvfj8fcEsIm1JJf7KDeIeYncybi71Fd/2iPFVUKJIUuXmPxfwXnn7eqzspQRNnS1zf+y8oe4V7Y6/DBD0vt+tMCjMCiaAfsyQshaje78hZbh0m3fWrdJMClpcTXAQIDAQABAoGAENG8JMXKT+FKJ4sRpdkxlWf4l+lXziv2vUF2rLtil+3XXFgdewbBdqgqF3EHvM/VzxKqTl2drgcKiLnJOvdYldq721OglkBfCEwI1yKjf/xWo/B2IuHsbkKrodBWLpzWhoYb8WLSePFPaYvIyRUxAiDSTUcySn1piHKh2L6dFj0CQQDTC5/3GZaFL1lKTp2/WOzp1a+W5pinNVYugQz6ZUWurIdkePiJswCeERa7IHnAfa8iHYWZhn2Y9jubl1W+aLgXAkEAwJRu+JFpW4AtiwRGRHpTLMGJh40eUI668RwXiBePLUFdWJMPsuZC//fWRLIZQ/Ukv5XLcFdQeAPh8crVQGlApwJAXSy2tRtg7vAWlc3bq00RW7Nx0EeCgd/0apejKTFo8FNPezZFVFXpIeAdjwQpfKiAl6k9AKj17oBXlLvdqTEGhQJBALgsPIST7EKJrwSILftHUUw4OyLbnuZD2hzEVOzeOxt4q6EN47Gf7OuHRe+ks+z+AQsIYuspVdexPuBSrudOwXkCQQDEdI7texImU7o6tXAt9mmVyVik9ibRaTnpbJJh+ox+EwYAMfQ7HqW8el3XH+q5tNNDNuR+voIuWJPRD30nOgb5";
    private X509Certificate entityCert;
    private String entityCertBase64 = "MIICvzCCAiigAwIBAgIJALQ1JXkgPO25MA0GCSqGSIb3DQEBBQUAMEoxCzAJBgNVBAYTAkNIMQ8wDQYDVQQIEwZadXJpY2gxFDASBgNVBAoTC2V4YW1wbGUub3JnMRQwEgYDVQQDEwtleGFtcGxlLm9yZzAeFw0wODEyMDQwNzUzNDBaFw0wOTEyMDQwNzUzNDBaMEoxCzAJBgNVBAYTAkNIMQ8wDQYDVQQIEwZadXJpY2gxFDASBgNVBAoTC2V4YW1wbGUub3JnMRQwEgYDVQQDEwtleGFtcGxlLm9yZzCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAnsMWLbfrRBZYcu/uVd5HQhgwE4olz/wNK9kPmpro7g89Gmrr34/H3BLCJtSSX+yg3iHmJ3Mm4u9RXf9ojxVVCiSFLl5j8X8F55+3qs7KUETZ0tc3/svKHuFe2OvwwQ9L7frTAozAomgH7MkLIWo3u/IWW4dJt31q3STApaXE1wECAwEAAaOBrDCBqTAdBgNVHQ4EFgQU0lf1wYwRJhvGZYL2WpMOykDNdeUwegYDVR0jBHMwcYAU0lf1wYwRJhvGZYL2WpMOykDNdeWhTqRMMEoxCzAJBgNVBAYTAkNIMQ8wDQYDVQQIEwZadXJpY2gxFDASBgNVBAoTC2V4YW1wbGUub3JnMRQwEgYDVQQDEwtleGFtcGxlLm9yZ4IJALQ1JXkgPO25MAwGA1UdEwQFMAMBAf8wDQYJKoZIhvcNAQEFBQADgYEAlhsuXNm5WMq7mILnbS+Xr+oi/LVezr4Yju+Qdh9AhYwbDaXnsZITHiAmfYhO5nTjstWMAHc6JZs7h8wDvqY92RvLY+Vx78MoJXIwqqLFH4oHm2UKpvsNivrNfD/qWPiKEYrXVVkDXUVA2yKupX1VtCru8kaJ42kAlCN9Bg4wezU=";
    private X509Certificate entityCert3AltNamesDNS_URL_IP;
    private String entityCert3AltNamesDNS_URL_IPBase64 = "MIIDzjCCAragAwIBAgIBMTANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTE4MjM0MFoXDTE3MDUxODE4MjM0MFowMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgfQwgfEwCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMEAGA1UdEQQ5MDeCEmFzaW1vdi5leGFtcGxlLm9yZ4YbaHR0cDovL2hlaW5sZWluLmV4YW1wbGUub3JnhwQKAQIDMA0GCSqGSIb3DQEBBQUAA4IBAQBLiDMyQ60ldIytVO1GCpp1S1sKJyTF56GVxHh/82hiRFbyPu+2eSl7UcJfH4ZNbAfHL1vDKTRJ9zoD8WRzpOCUtT0IPIA/Ex+8lFzZmujO10j3TMpp8Ii6+auYwi/Tosrfw1YCxF+GI5KO49CfDRr6yxUbMhbTN+ssK4UzFf36UbkeJ3EfDwB0WU70jnlkyO8f97X6mLd5QvRcwlkDMftP4+MB+inTlxDZ/w8NLXQoDW6p/8r91bupXe0xwuyEvow2xjxlzVcux2BZsUZYjBa07ZmNNBtF7WaQqH7l2OBCAdnBhvme5i/e0LK3Ivys+hcVyvCXs5XtFTFWDAVYvzQ6";
    private X509Certificate entityCert3AltNamesDNS_URN_IP;
    private String entityCert3AltNamesDNS_URN_IPBase64 = "MIIDyjCCArKgAwIBAgIBLDANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTA0NDQzOVoXDTE3MDUxODA0NDQzOVowMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgfAwge0wCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMDwGA1UdEQQ1MDOCEmFzaW1vdi5leGFtcGxlLm9yZ4YXdXJuOmZvbzpleGFtcGxlLm9yZzppZHCHBAoBAgMwDQYJKoZIhvcNAQEFBQADggEBAH7LRnOWJbP5p50lLvBaW6G0593OMChQIXVim9kf6Um4HQjC8/3BZPltyNMxn+xtUnRYAaKPDjbpr0CkM5lggJd8Q69XJiPTch9UQlcX+Ry7CXV+GsTnn6kgE5IW0ULqrp/ivVQVu6Af/dBS1+K+TddYOatNnABLr0lco5ppZ4v9HFIsoLljTrkdW4XrlYmW1Hx0SUVrYsbv2uRP3n1jEEYldvZOdhEGoEADSt46zE+HCG/ytfTYSDyola6OErB09e/oFDzzWGsOve69UV11bdeFgaMQJYloFHXq9MRKOCaKQLWxjwMd1MRJLJX6WpwZS600t2pJYMLFu19LDRfgX4M=";
    private X509Certificate entityCert1AltNameDNS;
    private String entityCert1AltNameDNSBase64 = "MIIDqzCCApOgAwIBAgIBLTANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTE3MzM0M1oXDTE3MDUxODE3MzM0M1owMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgdEwgc4wCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMB0GA1UdEQQWMBSCEmFzaW1vdi5leGFtcGxlLm9yZzANBgkqhkiG9w0BAQUFAAOCAQEAjSRiOpLAbrxkqQ0Yh+mUWCVA2ChSDBnFFDe4a3Z/87Tw7QEzU6U1xejCH6kGGZSmHLBMPLg31+QNiWwXDnQTqa8w/16oncUuw3olIK+C/r+F5uhakJcPq6LK8ZhSDi85YGMn1vPHP8FsC9/HMZS0Y/ouzDeZYwXc9ZwF8uMxh+vnKWUbyVDGuoTI4x0SIMgyrA917xpSG/1m9lJVVvF9S6/+n+ZpkIhpmvmOHGNicBoXsNk3tgHPzGTkn/DDx9SGmBUfyBEOTwlDHX36zqGRozWRVqGVYMb58L7dxLjnWkO50eVKajcKvJ1zBowSoiDQ50drULm5FSVzix3gUO1p6g==";
    private X509Certificate entityCert1AltNameURN;
    private String entityCert1AltNameURNBase64 = "MIIDsDCCApigAwIBAgIBLjANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTE3NDYyNVoXDTE3MDUxODE3NDYyNVowMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgdYwgdMwCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMCIGA1UdEQQbMBmGF3Vybjpmb286ZXhhbXBsZS5vcmc6aWRwMA0GCSqGSIb3DQEBBQUAA4IBAQA6REOOby69uy/zvgidjEuZRK/oacIKvjVm+1K0HSKbGdroCHRRMQS6s5IGRE2ef+wiwus1367/crxYEqa+Tu9iewyVNFkZjWm9ra+TkgoghA5DteoC0tYzUhWooWhA6FW7Ktn8yAdmGPV+bhMTwnrm9DiM9mAZr0Ew8qP78HWziw2qWM48LhdfuO2kiWzvinRx1wqKJjur9nY9piUOO32aTlzXZy2yLiOYVKUw2dKdxMmvwYxNYCEzNx2ERmDSbHoNZLn75WidNTnHpkn0rBh2J9ZS8j2swyoVoVp3rQRHDSQ9CJCNKVXWh/WnjgqnLpBzXKCLv/zrQ3t47OL2Jyso";
    private X509Certificate entityCert1AltNameURL;
    private String entityCert1AltNameURLBase64 = "MIIDtDCCApygAwIBAgIBMDANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTE4MTMwOFoXDTE3MDUxODE4MTMwOFowMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgdowgdcwCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMCYGA1UdEQQfMB2GG2h0dHA6Ly9oZWlubGVpbi5leGFtcGxlLm9yZzANBgkqhkiG9w0BAQUFAAOCAQEAQRJHMwtHvzdaTKg/GdSdt1u6H+tkspYESeDOFS0Ni9bm2nPrKLPHzWwVFriMwqtWT0ik7Sx8TK1jA2q3Wxgj+xS9kAvFtGyypq1HEMdVXwcQlyopSZEd3Oi7Bfam6eSy1ehVKkEwG9pry+0v6I1Z3gShPHBm/TcjEV3FIv6CTYgW9jZIBPKfI54xyQ7Ef07V608S6lpPGEOmjZPccQmiqu2fXTvmSxmDeXUY9lfn7SR3afmHOeDuovoa+sPZnyBmtsWcllmI328ZkSukaOXhLDLFLt2UA55Luy4/1cWTxEqyuizzTvjbHvvw7HF4/yBkNggcumQqr9gWqxNvvXFsNw==";
    private X509Certificate entityCert1AltNameIP;
    private String entityCert1AltNameIPBase64 = "MIIDnTCCAoWgAwIBAgIBLzANBgkqhkiG9w0BAQUFADAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnMB4XDTA3MDUyMTE3NDgwMloXDTE3MDUxODE3NDgwMlowMTESMBAGA1UEChMJSW50ZXJuZXQyMRswGQYDVQQDExJmb29iYXIuZXhhbXBsZS5vcmcwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDNWnkFmhy1vYa6gN/xBRKkZxFy3sUq2V0LsYb6Q3pe9Qlb6+BzaM5DrN8uIqqroBE3Wp0LtrgKuQTpDpNFBdS2p5afiUtOYLWBDtizTOzs3Z36MGMjIPUYQ4s03IP3yPh2ud6EKpDPiYqzNbkRaiIwmYSit5r+RMYvd6fuKvTOn6h7PZI5AD7Rda7VWh5OVSoZXlRx3qxFho+mZhW0q4fUfTi5lWwf4EhkfBlzgw/k5gf4cOi6rrGpRS1zxmbtX1RAg+I20z6d04g0N2WsK5stszgYKoIROJCiXwjraa8/SoFcILolWQpttVHBIUYlyDlm8mIFleZf4ReFpfm+nUYxAgMBAAGjgcMwgcAwCQYDVR0TBAIwADAsBglghkgBhvhCAQ0EHxYdT3BlblNTTCBHZW5lcmF0ZWQgQ2VydGlmaWNhdGUwHQYDVR0OBBYEFDgRgTkjaKoK6DoZfUZ4g9LDJUWuMFUGA1UdIwROMEyAFNXuZVPeUdqHrULqQW7yr9buRpQLoTGkLzAtMRIwEAYDVQQKEwlJbnRlcm5ldDIxFzAVBgNVBAMTDmNhLmV4YW1wbGUub3JnggEBMA8GA1UdEQQIMAaHBAoBAgMwDQYJKoZIhvcNAQEFBQADggEBAIgpJnJ9Pid+ldf/jvO/BRQkHdRkuzMP3AwLvzSIJPcJAw4Dvzqm57VQaJDnfqqXSN9POAPlpsBzBE8Xdtpp5TemJt7X2wjuCHTlvGY/HaPPvb3QielWsU4As6Xdk1xYovTPtGnbh+gsPT5jdrA+d5PKEsXicZEVqGOIRVINuDUhZsl0Y26SJmskWNKAb7l47jPQj8U2kkWUEWXkOv5FsyiB2KdxYGbJSpGwGLRWZNDbuVUjnuzQ29EWWbNwHxTbGMRjrI9Q4WynZ2IOcnG1hMjCU6L4uk4JfryIw4IBHGa8uUtskHqJ7TFJ/4taWyV/UB0djqOPjMACQpMBhEVRSBU=";
    private String altNameDNS;
    private String altNameURN;
    private String altNameURL;
    private String altNameIP;
    private Integer altNameTypeDNS;
    private Integer altNameTypeURI;
    private Integer altNameTypeIP;
    private static String subjectAltNameExtensionOID = "2.5.29.17";
    private String certPEM = "/data/certificate.pem";
    private String certDER = "/data/certificate.der";
    private String crlPEM = "/data/crl.pem";
    private String crlDER = "/data/crl.der";
    private String certEC = "/data/ec-certificate.pem";
    private String keyEC = "/data/ec-privkey-nopass.pem";

    @BeforeMethod
    protected void setUp() throws Exception {
        this.entityPrivateKey = KeySupport.buildJavaRSAPrivateKey((String)this.entityPrivKeyBase64);
        this.entityCert = X509Support.decodeCertificate((String)this.entityCertBase64);
        this.entityCert3AltNamesDNS_URL_IP = X509Support.decodeCertificate((String)this.entityCert3AltNamesDNS_URL_IPBase64);
        this.entityCert3AltNamesDNS_URN_IP = X509Support.decodeCertificate((String)this.entityCert3AltNamesDNS_URN_IPBase64);
        this.entityCert1AltNameDNS = X509Support.decodeCertificate((String)this.entityCert1AltNameDNSBase64);
        this.entityCert1AltNameURN = X509Support.decodeCertificate((String)this.entityCert1AltNameURNBase64);
        this.entityCert1AltNameURL = X509Support.decodeCertificate((String)this.entityCert1AltNameURLBase64);
        this.entityCert1AltNameIP = X509Support.decodeCertificate((String)this.entityCert1AltNameIPBase64);
        this.altNameDNS = "asimov.example.org";
        this.altNameURN = "urn:foo:example.org:idp";
        this.altNameURL = "http://heinlein.example.org";
        this.altNameIP = "10.1.2.3";
        this.altNameTypeIP = X509Support.IP_ADDRESS_ALT_NAME;
        this.altNameTypeURI = X509Support.URI_ALT_NAME;
        this.altNameTypeDNS = X509Support.DNS_ALT_NAME;
    }

    @Test
    public void testGetCommonNames() {
        List commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)1);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org, o=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)1);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org, cn=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)2);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        Assert.assertEquals((String)((String)commonNames.get(1)), (String)"MyOrg");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org, cn=WebServers, cn=Hosts, o=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)3);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        Assert.assertEquals((String)((String)commonNames.get(1)), (String)"WebServers");
        Assert.assertEquals((String)((String)commonNames.get(2)), (String)"Hosts");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("uid=foo, cn=Admins, cn=People, o=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)2);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"Admins");
        Assert.assertEquals((String)((String)commonNames.get(1)), (String)"People");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org+cn=bar.example.org+cn=baz.example.org, o=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)3);
        Assert.assertTrue((boolean)commonNames.contains("foo.example.org"));
        Assert.assertTrue((boolean)commonNames.contains("bar.example.org"));
        Assert.assertTrue((boolean)commonNames.contains("baz.example.org"));
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org+cn=bar.example.org+cn=baz.example.org, cn=Org1+cn=Org2"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)5);
        Assert.assertTrue((boolean)commonNames.contains("foo.example.org"));
        Assert.assertTrue((boolean)commonNames.contains("bar.example.org"));
        Assert.assertTrue((boolean)commonNames.contains("baz.example.org"));
        Assert.assertTrue((boolean)commonNames.contains("Org1"));
        Assert.assertTrue((boolean)commonNames.contains("Org2"));
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("uid=foo, o=MyOrg"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)0);
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("2.5.4.3=foo.example.org"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)1);
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org, o=foo \\,cn=www.apache.org"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)1);
        Assert.assertFalse((boolean)commonNames.contains("www.apache.org"));
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
        commonNames = X509Support.getCommonNames((X500Principal)new X500Principal("cn=foo.example.org, o=cn=www.apache.org\\, foo"));
        Assert.assertNotNull((Object)commonNames);
        Assert.assertEquals((int)commonNames.size(), (int)1);
        Assert.assertFalse((boolean)commonNames.contains("www.apache.org"));
        Assert.assertEquals((String)((String)commonNames.get(0)), (String)"foo.example.org");
    }

    @Test
    public void testGetSubjectKeyIdentifier() throws DecoderException {
        String hexSKI = "D2:57:F5:C1:8C:11:26:1B:C6:65:82:F6:5A:93:0E:CA:40:CD:75:E5";
        byte[] controlSKI = Hex.decodeHex((char[])hexSKI.replaceAll(":", "").toCharArray());
        byte[] certSKI = X509Support.getSubjectKeyIdentifier((X509Certificate)this.entityCert);
        Assert.assertEquals((byte[])certSKI, (byte[])controlSKI);
    }

    @Test
    public void testDetermineEntityCertificate() throws Exception {
        ArrayList<X509Certificate> certs = new ArrayList<X509Certificate>();
        certs.add(this.entityCert3AltNamesDNS_URL_IP);
        certs.add(this.entityCert1AltNameDNS);
        certs.add(this.entityCert);
        Assert.assertTrue((boolean)X509Support.determineEntityCertificate(certs, (PrivateKey)this.entityPrivateKey).equals(this.entityCert));
    }

    @Test
    public void testGetSubjectAltNames1NameDNS() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert1AltNameDNS;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeDNS);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameDNS), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testGetSubjectAltNames1NameURN() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert1AltNameURN;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeURI);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameURN), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testGetSubjectAltNames1NameURL() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert1AltNameURL;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeURI);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameURL), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testGetSubjectAltNames1NameIP() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert1AltNameIP;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeIP);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameIP), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testGetSubjectAltNames3NamesDNS_URL_IP() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert3AltNamesDNS_URL_IP;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeDNS);
        nameTypes.add(this.altNameTypeURI);
        nameTypes.add(this.altNameTypeIP);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameDNS), (String)"Failed to find expected KeyName value");
        Assert.assertTrue((boolean)altNames.contains(this.altNameURL), (String)"Failed to find expected KeyName value");
        Assert.assertTrue((boolean)altNames.contains(this.altNameIP), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testGetSubjectAltNames3NamesDNS_URN_IP() throws SecurityException, CertificateParsingException {
        X509Certificate cert = this.entityCert3AltNamesDNS_URN_IP;
        byte[] extensionValue = cert.getExtensionValue(subjectAltNameExtensionOID);
        Assert.assertNotNull((Object)extensionValue, (String)"Entity cert's Java native getExtensionValue() was null");
        Assert.assertTrue((extensionValue.length > 0 ? 1 : 0) != 0, (String)"Entity cert's extension value was empty");
        HashSet<Integer> nameTypes = new HashSet<Integer>();
        nameTypes.add(this.altNameTypeDNS);
        nameTypes.add(this.altNameTypeURI);
        nameTypes.add(this.altNameTypeIP);
        List altNames = this.getAltNames(cert, nameTypes);
        Assert.assertNotNull((Object)altNames, (String)"X509Support.getAltNames() returned null");
        Assert.assertTrue((boolean)altNames.contains(this.altNameDNS), (String)"Failed to find expected KeyName value");
        Assert.assertTrue((boolean)altNames.contains(this.altNameURN), (String)"Failed to find expected KeyName value");
        Assert.assertTrue((boolean)altNames.contains(this.altNameIP), (String)"Failed to find expected KeyName value");
    }

    @Test
    public void testDecodeCertPEM() throws Exception {
        InputStream certInS = X509SupportTest.class.getResourceAsStream(this.certPEM);
        byte[] certBytes = new byte[certInS.available()];
        certInS.read(certBytes);
        Collection certs = X509Support.decodeCertificates((byte[])certBytes);
        Assert.assertNotNull((Object)certs);
        Assert.assertEquals((int)certs.size(), (int)2);
    }

    @Test
    public void testDecodeCertDER() throws Exception {
        InputStream certInS = X509SupportTest.class.getResourceAsStream(this.certDER);
        byte[] certBytes = new byte[certInS.available()];
        certInS.read(certBytes);
        Collection certs = X509Support.decodeCertificates((byte[])certBytes);
        Assert.assertNotNull((Object)certs);
        Assert.assertEquals((int)certs.size(), (int)1);
        X509Certificate cert = X509Support.decodeCertificate((byte[])certBytes);
        Assert.assertNotNull((Object)cert);
    }

    @Test
    public void testDecodeCRLPEM() throws Exception {
        InputStream crlInS = X509SupportTest.class.getResourceAsStream(this.crlPEM);
        byte[] crlBytes = new byte[crlInS.available()];
        crlInS.read(crlBytes);
        Collection crls = X509Support.decodeCRLs((byte[])crlBytes);
        Assert.assertNotNull((Object)crls);
        Assert.assertEquals((int)crls.size(), (int)1);
    }

    @Test
    public void testDecodeCRLDER() throws Exception {
        InputStream crlInS = X509SupportTest.class.getResourceAsStream(this.crlDER);
        byte[] crlBytes = new byte[crlInS.available()];
        crlInS.read(crlBytes);
        Collection crls = X509Support.decodeCRLs((byte[])crlBytes);
        Assert.assertNotNull((Object)crls);
        Assert.assertEquals((int)crls.size(), (int)1);
    }

    @Test(enabled=false)
    public void testEC() throws Exception {
        InputStream certInS = X509SupportTest.class.getResourceAsStream(this.certEC);
        byte[] certBytes = new byte[certInS.available()];
        certInS.read(certBytes);
        Collection certs = X509Support.decodeCertificates((byte[])certBytes);
        Assert.assertNotNull((Object)certs);
        Assert.assertEquals((int)certs.size(), (int)1);
        PublicKey pubkey = ((X509Certificate)certs.iterator().next()).getPublicKey();
        Assert.assertNotNull((Object)pubkey);
        PrivateKey key = KeySupport.decodePrivateKey((InputStream)KeySupportTest.class.getResourceAsStream(this.keyEC), null);
        Assert.assertNotNull((Object)key);
        Assert.assertTrue((boolean)KeySupport.matchKeyPair((PublicKey)pubkey, (PrivateKey)key));
    }

    private List getAltNames(X509Certificate cert, Set<Integer> nameTypes) {
        Integer[] array = new Integer[nameTypes.size()];
        nameTypes.toArray(array);
        return X509Support.getAltNames((X509Certificate)cert, (Integer[])array);
    }
}

