/*
 * Decompiled with CFR 0.152.
 */
package org.opensaml.xmlsec.keyinfo;

import com.google.common.base.Strings;
import java.math.BigInteger;
import java.security.KeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.cert.CRLException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.interfaces.DSAParams;
import java.security.interfaces.DSAPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.List;
import net.shibboleth.utilities.java.support.codec.Base64Support;
import net.shibboleth.utilities.java.support.codec.DecodingException;
import net.shibboleth.utilities.java.support.codec.EncodingException;
import org.opensaml.core.xml.XMLObjectBaseTestCase;
import org.opensaml.security.SecurityException;
import org.opensaml.security.crypto.KeySupport;
import org.opensaml.security.x509.X509Support;
import org.opensaml.xmlsec.keyinfo.KeyInfoSupport;
import org.opensaml.xmlsec.signature.DEREncodedKeyValue;
import org.opensaml.xmlsec.signature.DSAKeyValue;
import org.opensaml.xmlsec.signature.Exponent;
import org.opensaml.xmlsec.signature.G;
import org.opensaml.xmlsec.signature.KeyInfo;
import org.opensaml.xmlsec.signature.KeyValue;
import org.opensaml.xmlsec.signature.Modulus;
import org.opensaml.xmlsec.signature.P;
import org.opensaml.xmlsec.signature.Q;
import org.opensaml.xmlsec.signature.RSAKeyValue;
import org.opensaml.xmlsec.signature.X509CRL;
import org.opensaml.xmlsec.signature.X509Certificate;
import org.opensaml.xmlsec.signature.X509Data;
import org.opensaml.xmlsec.signature.X509Digest;
import org.opensaml.xmlsec.signature.X509IssuerSerial;
import org.opensaml.xmlsec.signature.X509SKI;
import org.opensaml.xmlsec.signature.X509SubjectName;
import org.opensaml.xmlsec.signature.Y;
import org.testng.Assert;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;

public class KeyInfoSupportTest
extends XMLObjectBaseTestCase {
    private final String certNoExtensions = "MIIBwjCCASugAwIBAgIJAMrW6QSeKNBJMA0GCSqGSIb3DQEBBAUAMCMxITAfBgNVBAMTGG5vZXh0ZW5zaW9ucy5leGFtcGxlLm9yZzAeFw0wNzA1MTkxNzU2NTVaFw0wNzA2MTgxNzU2NTVaMCMxITAfBgNVBAMTGG5vZXh0ZW5zaW9ucy5leGFtcGxlLm9yZzCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAw8xxu6TLqEdmnyXVZjiUoRHN6yHyobZaRK+tBEmWkD4nTlOVmTKWBCO/F4OnugaJbSTH+7Jk37l8/XYYBSIkW0+L2BglzQ2JCux/uoRu146QDIk9f5PIFs+Fxy7VRVUUZiOsonB/PNVqA7OVbPxzr1SKPSE0s9CHaDjCaEs2BnMCAwEAATANBgkqhkiG9w0BAQQFAAOBgQAuI/l80wb8K6RT1EKrAcfr9JAlJR4jmVnCK7j3Ulx++U98ze2G6/cluLxrbnqwXmxJNC3nt6xkQVJUX1UFg+zkmRrst2Nv8TTrR7S30az068BHfrZLRSUConG9jXXj+hJq+w/ojmrq8MzvJSczkA2BvsEUBARYo53na7RMgk+xWg==";
    private final String cert1SubjectDN = "CN=Basic Self-Issued New Key CA,O=Test Certificates,C=US";
    private final String cert1SKIPlainBase64 = "r7n5HcJFGMy4IeKnR7xJvRm1eCg=";
    private final String cert1DigestBase64 = "EmkP8ttMw28A/JoA3KcO11eez7Q=";
    private final String cert1 = "MIICgjCCAeugAwIBAgIBEzANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMFAxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczElMCMGA1UEAxMcQmFzaWMgU2VsZi1Jc3N1ZWQgTmV3IEtleSBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAtCkygqcMEOy3i8p6ZV3685us1lOugSU4pUMRJNRH/lV2ykesk+JRcQy1s7WS12j9GCnSJ919/TgeKLmV3ps1fC1B8HziC0mzBAr+7f5LkJqSf0kS0kfpyLOoO8VSJCip/8uENkSkpvX+Lak96OKzhtyvi4KpUdQKfwpg6xUqakECAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFK+5+R3CRRjMuCHip0e8Sb0ZtXgoMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCuRBfDy2gSPp2kZR7OAvt+xDx4toJ9ImImUvJ94AOLd6Uxsi2dvQT5HLrIBrTYsSfQj1pA50XY2F7k3eM/+JhYCcyZD9XtAslpOkjwACPJnODFAY8PWC00CcOxGb6q+S/VkrCwvlBeMjevIH4bHvAymWsZndBZhcG8gBmDrZMwhQ==";
    private final String cert2SubjectDN = "CN=Good CA,O=Test Certificates,C=US";
    private final String cert2 = "MIICbTCCAdagAwIBAgIBAjANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMDsxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczEQMA4GA1UEAxMHR29vZCBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEArsI1lQuXKwOxSkOVRaPwlhMQtgp0p7HT4rKLGqojfY0twvMDc4rC9uj97wlh98kkraMx3r0wlllYSQ+Cp9mCCNu/C/Y2IbZCyG+io4A3Um3q/QGvbHlclmrJb0j0MQi3o88GhE8Q6Vy6SGwFXGpKDJMpLSFpPxz8lh7M6J56Ex8CAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFLcupoLLwsi8qHsnRNc1M9+aFZTHMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCOls9+0kEUS71w+KoQhfkVLdAKANXUmGCVZHL1zsyacPP/Q8IsCNvwjefZpgc0cuhtnHt2uDd0/zYLRmgcvJwfx5vwOfmDN13mMB8Za+cg3sZ/NI8MqQseKvS3fWqXaK6FJoKLzxId0iUGntbF4c5+rPFArzqM6IE7f9cMD5FqrA==";
    private final String crl1IssuerDN = "CN=Basic Self-Issued CRL Signing Key CA,O=Test Certificates,C=US";
    private final String crl1 = "MIIBdTCB3wIBATANBgkqhkiG9w0BAQUFADBYMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxLTArBgNVBAMTJEJhc2ljIFNlbGYtSXNzdWVkIENSTCBTaWduaW5nIEtleSBDQRcNMDEwNDE5MTQ1NzIwWhcNMTEwNDE5MTQ1NzIwWjAiMCACAQMXDTAxMDQxOTE0NTcyMFowDDAKBgNVHRUEAwoBAaAvMC0wHwYDVR0jBBgwFoAUD3LKM0OpxBFRq2PaRIcPYaT0vkcwCgYDVR0UBAMCAQEwDQYJKoZIhvcNAQEFBQADgYEAXM2Poz2eZPdkc5wsOeLn1w64HD6bHRTcmMKOWh/lRzH9fqfVn1IxyBD30KKEP3fH8bp+JGKtBa4ce//w4s5V9SfTzCR/yB2muM5CBeEG7B+HTNVpjXhZ0jOUHDsnaIA9bz2mx58rOZ/Xw4Prd73Mf5azrSRomdEavwUcjD4qAvg=";
    private final String rsaPubKey1 = "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAw/WnsbA8frhQ+8EoPgMrQjpINjt20U/MvsvmoAgQnAgEF4OYt9Vj9/2YvMO4NvX1fNDFzoYRyOMrypF7skAPcITUhdcPSEpI4nsf5yFZLziK/tQ26RsccE7WhpGB8eHu9tfseelgyioorvmt+JCoP15c5rYUuIfVC+eEsYolw344q6N61OACHETuySL0a1+GFu3WoISXte1pQIst7HKvBbHH41HEWAxT6e0hlD5PyKL4lBJadGHXg8Zz4r2jV2n6+Ox7raEWmtVCGFxsAoCRalu6nvs2++5Nnb4C1SE640esfYhfeMd5JYfsTNMaQ8sZLpsWdglAGpa/Q87K19LIwwIDAQAB";
    private final String dsaPubKey1 = "MIIDOjCCAi0GByqGSM44BAEwggIgAoIBAQCWV7IK073aK2C3yggy69qXkxCw30j5Ig0s1/GHgq5jEZf8FTGVpehX5qaYlRC3TBMSN4WAgkG+nFnsjHb6kIYkayV8ZVvIIgEBCeaZg016f90G+Rre5C38G3OwsODKjPsVZCV5YQ9rm6lWMOfMRSUzJuFA0fdxRLssAfKLI5JmzupliO2iH5FU3+dQr0UvcPwPjjRDA9JIi3ShKdmq9f/SzRM9AJPssjc0v4lRVMKWkTHLjbRH2XiOxsok/oL7NVTJ9hvd3xqi1/O3MM2pNhYaQoA0kLqqsr006dNftgo8n/zrBFMC6iP7tmxhuRxgXXkNo5xiQCvAX7HsGno4y9ilAhUAjKlvCQhbGeQo3fWbwVJMdokSK5ECggEAfERqa+S8UwjuvNxGlisuBGzR7IqqHSQ0cjFIBD61CkYh0k0Y9am6ZL2jiAkRICdkW6f9lmGy0HidCwC56WeAYpLyfJslBAjC4r0t6U8a822fECVcbsPNLDULoQG0KjVRtYfFH5GedNQ8LRkG8b+XIe4G74+vXOatVu8L9QXQKYx9diOAHx8ghpt1pC0UAqPzAgVGNWIPQ+VO7WEYOYuVw+/uFoHiaU1OZOTFC4VXk2+33AasT4i6It7DIESp+ye9lPnNU6nLEBNSnXdnBgaH27m8QnFRTfrjimiGBwBTQvbjequRvM5dExfUqyCd2BUOK1lbaQmjZnCMH6k3ZFiAYgOCAQUAAoIBAGnDwMuRoRGJHUhjjeePKwP9BgCc7dtKlB7QMnIHGPv03hdVPo9ezaQ5mFxdzQdXoLR2BFucDtSj1je3e5L9KEnHZ5fHnislBnzSvYR5V8LwTa5mbNS4VHkAv8Eh3WG9tp1S/f9ymefKHB7ISlskT7kODCIbr5HHU/n1zXtMRjoslY1A+nFlWiAaIvjnj/C8x0BWBkhuSKX/2PbljnmIdGV7mJK9/XUHnyKgZBxXEul2mlvGkrgUvyv+qYsCFsKSSrkB1Mj2Ql5xmTMaePMEmvOr6fDAP0OH8cvADEZjx0s/5vvoBFPGGmPrHJluEVS0Fu8I9sROg9YjyuhRV0b8xHo=";
    private X509Certificate xmlCert1;
    private X509Certificate xmlCert2;
    private X509CRL xmlCRL1;
    private X509Data xmlX509Data;
    private KeyInfo keyInfo;
    private KeyValue keyValue;
    private DSAKeyValue xmlDSAKeyValue1;
    private DSAKeyValue xmlDSAKeyValue1NoParams;
    private RSAKeyValue xmlRSAKeyValue1;
    private int numExpectedCerts;
    private int numExpectedCRLs;
    private java.security.cert.X509Certificate javaCert1;
    private java.security.cert.X509CRL javaCRL1;
    private RSAPublicKey javaRSAPubKey1;
    private DSAPublicKey javaDSAPubKey1;
    private DSAParams javaDSAParams1;

    @BeforeMethod
    protected void setUp() throws Exception {
        this.xmlCert1 = (X509Certificate)this.buildXMLObject(X509Certificate.DEFAULT_ELEMENT_NAME);
        this.xmlCert1.setValue("MIICgjCCAeugAwIBAgIBEzANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMFAxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczElMCMGA1UEAxMcQmFzaWMgU2VsZi1Jc3N1ZWQgTmV3IEtleSBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAtCkygqcMEOy3i8p6ZV3685us1lOugSU4pUMRJNRH/lV2ykesk+JRcQy1s7WS12j9GCnSJ919/TgeKLmV3ps1fC1B8HziC0mzBAr+7f5LkJqSf0kS0kfpyLOoO8VSJCip/8uENkSkpvX+Lak96OKzhtyvi4KpUdQKfwpg6xUqakECAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFK+5+R3CRRjMuCHip0e8Sb0ZtXgoMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCuRBfDy2gSPp2kZR7OAvt+xDx4toJ9ImImUvJ94AOLd6Uxsi2dvQT5HLrIBrTYsSfQj1pA50XY2F7k3eM/+JhYCcyZD9XtAslpOkjwACPJnODFAY8PWC00CcOxGb6q+S/VkrCwvlBeMjevIH4bHvAymWsZndBZhcG8gBmDrZMwhQ==");
        this.xmlCert2 = (X509Certificate)this.buildXMLObject(X509Certificate.DEFAULT_ELEMENT_NAME);
        this.xmlCert2.setValue("MIICbTCCAdagAwIBAgIBAjANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMDsxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczEQMA4GA1UEAxMHR29vZCBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEArsI1lQuXKwOxSkOVRaPwlhMQtgp0p7HT4rKLGqojfY0twvMDc4rC9uj97wlh98kkraMx3r0wlllYSQ+Cp9mCCNu/C/Y2IbZCyG+io4A3Um3q/QGvbHlclmrJb0j0MQi3o88GhE8Q6Vy6SGwFXGpKDJMpLSFpPxz8lh7M6J56Ex8CAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFLcupoLLwsi8qHsnRNc1M9+aFZTHMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCOls9+0kEUS71w+KoQhfkVLdAKANXUmGCVZHL1zsyacPP/Q8IsCNvwjefZpgc0cuhtnHt2uDd0/zYLRmgcvJwfx5vwOfmDN13mMB8Za+cg3sZ/NI8MqQseKvS3fWqXaK6FJoKLzxId0iUGntbF4c5+rPFArzqM6IE7f9cMD5FqrA==");
        this.xmlCRL1 = (X509CRL)this.buildXMLObject(X509CRL.DEFAULT_ELEMENT_NAME);
        this.xmlCRL1.setValue("MIIBdTCB3wIBATANBgkqhkiG9w0BAQUFADBYMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxLTArBgNVBAMTJEJhc2ljIFNlbGYtSXNzdWVkIENSTCBTaWduaW5nIEtleSBDQRcNMDEwNDE5MTQ1NzIwWhcNMTEwNDE5MTQ1NzIwWjAiMCACAQMXDTAxMDQxOTE0NTcyMFowDDAKBgNVHRUEAwoBAaAvMC0wHwYDVR0jBBgwFoAUD3LKM0OpxBFRq2PaRIcPYaT0vkcwCgYDVR0UBAMCAQEwDQYJKoZIhvcNAQEFBQADgYEAXM2Poz2eZPdkc5wsOeLn1w64HD6bHRTcmMKOWh/lRzH9fqfVn1IxyBD30KKEP3fH8bp+JGKtBa4ce//w4s5V9SfTzCR/yB2muM5CBeEG7B+HTNVpjXhZ0jOUHDsnaIA9bz2mx58rOZ/Xw4Prd73Mf5azrSRomdEavwUcjD4qAvg=");
        this.xmlX509Data = (X509Data)this.buildXMLObject(X509Data.DEFAULT_ELEMENT_NAME);
        this.xmlX509Data.getX509Certificates().add(this.xmlCert1);
        this.xmlX509Data.getX509Certificates().add(this.xmlCert2);
        this.xmlX509Data.getX509CRLs().add(this.xmlCRL1);
        this.keyValue = (KeyValue)this.buildXMLObject(KeyValue.DEFAULT_ELEMENT_NAME);
        this.keyInfo = (KeyInfo)this.buildXMLObject(KeyInfo.DEFAULT_ELEMENT_NAME);
        this.keyInfo.getX509Datas().add(this.xmlX509Data);
        this.numExpectedCerts = 2;
        this.numExpectedCRLs = 1;
        this.javaCert1 = X509Support.decodeCertificate((String)"MIICgjCCAeugAwIBAgIBEzANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMFAxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczElMCMGA1UEAxMcQmFzaWMgU2VsZi1Jc3N1ZWQgTmV3IEtleSBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAtCkygqcMEOy3i8p6ZV3685us1lOugSU4pUMRJNRH/lV2ykesk+JRcQy1s7WS12j9GCnSJ919/TgeKLmV3ps1fC1B8HziC0mzBAr+7f5LkJqSf0kS0kfpyLOoO8VSJCip/8uENkSkpvX+Lak96OKzhtyvi4KpUdQKfwpg6xUqakECAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFK+5+R3CRRjMuCHip0e8Sb0ZtXgoMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCuRBfDy2gSPp2kZR7OAvt+xDx4toJ9ImImUvJ94AOLd6Uxsi2dvQT5HLrIBrTYsSfQj1pA50XY2F7k3eM/+JhYCcyZD9XtAslpOkjwACPJnODFAY8PWC00CcOxGb6q+S/VkrCwvlBeMjevIH4bHvAymWsZndBZhcG8gBmDrZMwhQ==");
        X509Support.decodeCertificate((String)"MIICbTCCAdagAwIBAgIBAjANBgkqhkiG9w0BAQUFADBAMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxFTATBgNVBAMTDFRydXN0IEFuY2hvcjAeFw0wMTA0MTkxNDU3MjBaFw0xMTA0MTkxNDU3MjBaMDsxCzAJBgNVBAYTAlVTMRowGAYDVQQKExFUZXN0IENlcnRpZmljYXRlczEQMA4GA1UEAxMHR29vZCBDQTCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEArsI1lQuXKwOxSkOVRaPwlhMQtgp0p7HT4rKLGqojfY0twvMDc4rC9uj97wlh98kkraMx3r0wlllYSQ+Cp9mCCNu/C/Y2IbZCyG+io4A3Um3q/QGvbHlclmrJb0j0MQi3o88GhE8Q6Vy6SGwFXGpKDJMpLSFpPxz8lh7M6J56Ex8CAwEAAaN8MHowHwYDVR0jBBgwFoAU+2zULYGeyid6ng2wPOqavIf/SeowHQYDVR0OBBYEFLcupoLLwsi8qHsnRNc1M9+aFZTHMA4GA1UdDwEB/wQEAwIBBjAXBgNVHSAEEDAOMAwGCmCGSAFlAwIBMAEwDwYDVR0TAQH/BAUwAwEB/zANBgkqhkiG9w0BAQUFAAOBgQCOls9+0kEUS71w+KoQhfkVLdAKANXUmGCVZHL1zsyacPP/Q8IsCNvwjefZpgc0cuhtnHt2uDd0/zYLRmgcvJwfx5vwOfmDN13mMB8Za+cg3sZ/NI8MqQseKvS3fWqXaK6FJoKLzxId0iUGntbF4c5+rPFArzqM6IE7f9cMD5FqrA==");
        this.javaCRL1 = X509Support.decodeCRL((String)"MIIBdTCB3wIBATANBgkqhkiG9w0BAQUFADBYMQswCQYDVQQGEwJVUzEaMBgGA1UEChMRVGVzdCBDZXJ0aWZpY2F0ZXMxLTArBgNVBAMTJEJhc2ljIFNlbGYtSXNzdWVkIENSTCBTaWduaW5nIEtleSBDQRcNMDEwNDE5MTQ1NzIwWhcNMTEwNDE5MTQ1NzIwWjAiMCACAQMXDTAxMDQxOTE0NTcyMFowDDAKBgNVHRUEAwoBAaAvMC0wHwYDVR0jBBgwFoAUD3LKM0OpxBFRq2PaRIcPYaT0vkcwCgYDVR0UBAMCAQEwDQYJKoZIhvcNAQEFBQADgYEAXM2Poz2eZPdkc5wsOeLn1w64HD6bHRTcmMKOWh/lRzH9fqfVn1IxyBD30KKEP3fH8bp+JGKtBa4ce//w4s5V9SfTzCR/yB2muM5CBeEG7B+HTNVpjXhZ0jOUHDsnaIA9bz2mx58rOZ/Xw4Prd73Mf5azrSRomdEavwUcjD4qAvg=");
        this.javaDSAPubKey1 = KeySupport.buildJavaDSAPublicKey((String)"MIIDOjCCAi0GByqGSM44BAEwggIgAoIBAQCWV7IK073aK2C3yggy69qXkxCw30j5Ig0s1/GHgq5jEZf8FTGVpehX5qaYlRC3TBMSN4WAgkG+nFnsjHb6kIYkayV8ZVvIIgEBCeaZg016f90G+Rre5C38G3OwsODKjPsVZCV5YQ9rm6lWMOfMRSUzJuFA0fdxRLssAfKLI5JmzupliO2iH5FU3+dQr0UvcPwPjjRDA9JIi3ShKdmq9f/SzRM9AJPssjc0v4lRVMKWkTHLjbRH2XiOxsok/oL7NVTJ9hvd3xqi1/O3MM2pNhYaQoA0kLqqsr006dNftgo8n/zrBFMC6iP7tmxhuRxgXXkNo5xiQCvAX7HsGno4y9ilAhUAjKlvCQhbGeQo3fWbwVJMdokSK5ECggEAfERqa+S8UwjuvNxGlisuBGzR7IqqHSQ0cjFIBD61CkYh0k0Y9am6ZL2jiAkRICdkW6f9lmGy0HidCwC56WeAYpLyfJslBAjC4r0t6U8a822fECVcbsPNLDULoQG0KjVRtYfFH5GedNQ8LRkG8b+XIe4G74+vXOatVu8L9QXQKYx9diOAHx8ghpt1pC0UAqPzAgVGNWIPQ+VO7WEYOYuVw+/uFoHiaU1OZOTFC4VXk2+33AasT4i6It7DIESp+ye9lPnNU6nLEBNSnXdnBgaH27m8QnFRTfrjimiGBwBTQvbjequRvM5dExfUqyCd2BUOK1lbaQmjZnCMH6k3ZFiAYgOCAQUAAoIBAGnDwMuRoRGJHUhjjeePKwP9BgCc7dtKlB7QMnIHGPv03hdVPo9ezaQ5mFxdzQdXoLR2BFucDtSj1je3e5L9KEnHZ5fHnislBnzSvYR5V8LwTa5mbNS4VHkAv8Eh3WG9tp1S/f9ymefKHB7ISlskT7kODCIbr5HHU/n1zXtMRjoslY1A+nFlWiAaIvjnj/C8x0BWBkhuSKX/2PbljnmIdGV7mJK9/XUHnyKgZBxXEul2mlvGkrgUvyv+qYsCFsKSSrkB1Mj2Ql5xmTMaePMEmvOr6fDAP0OH8cvADEZjx0s/5vvoBFPGGmPrHJluEVS0Fu8I9sROg9YjyuhRV0b8xHo=");
        this.javaRSAPubKey1 = KeySupport.buildJavaRSAPublicKey((String)"MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAw/WnsbA8frhQ+8EoPgMrQjpINjt20U/MvsvmoAgQnAgEF4OYt9Vj9/2YvMO4NvX1fNDFzoYRyOMrypF7skAPcITUhdcPSEpI4nsf5yFZLziK/tQ26RsccE7WhpGB8eHu9tfseelgyioorvmt+JCoP15c5rYUuIfVC+eEsYolw344q6N61OACHETuySL0a1+GFu3WoISXte1pQIst7HKvBbHH41HEWAxT6e0hlD5PyKL4lBJadGHXg8Zz4r2jV2n6+Ox7raEWmtVCGFxsAoCRalu6nvs2++5Nnb4C1SE640esfYhfeMd5JYfsTNMaQ8sZLpsWdglAGpa/Q87K19LIwwIDAQAB");
        this.xmlRSAKeyValue1 = (RSAKeyValue)this.buildXMLObject(RSAKeyValue.DEFAULT_ELEMENT_NAME);
        Modulus modulus = (Modulus)this.buildXMLObject(Modulus.DEFAULT_ELEMENT_NAME);
        Exponent exponent = (Exponent)this.buildXMLObject(Exponent.DEFAULT_ELEMENT_NAME);
        modulus.setValueBigInt(this.javaRSAPubKey1.getModulus());
        exponent.setValueBigInt(this.javaRSAPubKey1.getPublicExponent());
        this.xmlRSAKeyValue1.setModulus(modulus);
        this.xmlRSAKeyValue1.setExponent(exponent);
        this.xmlDSAKeyValue1 = (DSAKeyValue)this.buildXMLObject(DSAKeyValue.DEFAULT_ELEMENT_NAME);
        P p = (P)this.buildXMLObject(P.DEFAULT_ELEMENT_NAME);
        Q q = (Q)this.buildXMLObject(Q.DEFAULT_ELEMENT_NAME);
        G g = (G)this.buildXMLObject(G.DEFAULT_ELEMENT_NAME);
        Y y1 = (Y)this.buildXMLObject(Y.DEFAULT_ELEMENT_NAME);
        p.setValueBigInt(this.javaDSAPubKey1.getParams().getP());
        q.setValueBigInt(this.javaDSAPubKey1.getParams().getQ());
        g.setValueBigInt(this.javaDSAPubKey1.getParams().getG());
        y1.setValueBigInt(this.javaDSAPubKey1.getY());
        this.xmlDSAKeyValue1.setP(p);
        this.xmlDSAKeyValue1.setQ(q);
        this.xmlDSAKeyValue1.setG(g);
        this.xmlDSAKeyValue1.setY(y1);
        this.xmlDSAKeyValue1NoParams = (DSAKeyValue)this.buildXMLObject(DSAKeyValue.DEFAULT_ELEMENT_NAME);
        Y y2 = (Y)this.buildXMLObject(Y.DEFAULT_ELEMENT_NAME);
        y2.setValueBigInt(this.javaDSAPubKey1.getY());
        this.xmlDSAKeyValue1NoParams.setY(y2);
        this.javaDSAParams1 = this.javaDSAPubKey1.getParams();
    }

    @Test
    public void testCertConversionXMLtoJava() throws CertificateException {
        java.security.cert.X509Certificate javaCert = null;
        try {
            javaCert = KeyInfoSupport.getCertificate((X509Certificate)this.xmlCert1);
        }
        catch (CertificateException e) {
            Assert.fail((String)("Conversion from XML X509Certificate format to java.security.cert.X509Certificate failed: " + e));
        }
        Assert.assertNotNull((Object)javaCert, (String)"Cert1 was null, failed to convert from XML to Java representation");
        Assert.assertEquals((String)javaCert.getSubjectX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued New Key CA,O=Test Certificates,C=US", (String)"Cert1 SubjectDN");
        Assert.assertEquals((Object)javaCert, (Object)X509Support.decodeCertificate((String)this.xmlCert1.getValue()), (String)"Java cert was not the expected value");
        List javaCertList = null;
        try {
            javaCertList = KeyInfoSupport.getCertificates((X509Data)this.xmlX509Data);
        }
        catch (CertificateException e) {
            Assert.fail((String)("Obtaining certs from X509Data failed: " + e));
        }
        Assert.assertEquals((int)javaCertList.size(), (int)this.numExpectedCerts, (String)"# of certs returned");
        Assert.assertEquals((String)((java.security.cert.X509Certificate)javaCertList.get(0)).getSubjectX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued New Key CA,O=Test Certificates,C=US", (String)"Cert1 SubjectDN");
        Assert.assertEquals((String)((java.security.cert.X509Certificate)javaCertList.get(1)).getSubjectX500Principal().getName("RFC2253"), (String)"CN=Good CA,O=Test Certificates,C=US", (String)"Cert2 SubjectDN");
        try {
            javaCertList = KeyInfoSupport.getCertificates((KeyInfo)this.keyInfo);
        }
        catch (CertificateException e) {
            Assert.fail((String)("Obtaining certs from KeyInfo failed: " + e));
        }
        Assert.assertEquals((int)javaCertList.size(), (int)this.numExpectedCerts, (String)"# of certs returned");
        Assert.assertEquals((String)((java.security.cert.X509Certificate)javaCertList.get(0)).getSubjectX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued New Key CA,O=Test Certificates,C=US", (String)"Cert1 SubjectDN");
        Assert.assertEquals((String)((java.security.cert.X509Certificate)javaCertList.get(1)).getSubjectX500Principal().getName("RFC2253"), (String)"CN=Good CA,O=Test Certificates,C=US", (String)"Cert2 SubjectDN");
    }

    @Test
    public void testCRLConversionXMLtoJava() throws CertificateException, CRLException {
        java.security.cert.X509CRL javaCRL = null;
        try {
            javaCRL = KeyInfoSupport.getCRL((X509CRL)this.xmlCRL1);
        }
        catch (CRLException e) {
            Assert.fail((String)("Conversion from XML X509CRL format to java.security.cert.X509CRL failed: " + e));
        }
        Assert.assertNotNull((Object)javaCRL, (String)"CRL was null, failed to convert from XML to Java representation");
        Assert.assertEquals((String)javaCRL.getIssuerX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued CRL Signing Key CA,O=Test Certificates,C=US", (String)"CRL IssuerDN");
        Assert.assertEquals((Object)javaCRL, (Object)X509Support.decodeCRL((String)this.xmlCRL1.getValue()), (String)"Java CRL was not the expected value");
        List javaCRLList = null;
        try {
            javaCRLList = KeyInfoSupport.getCRLs((X509Data)this.xmlX509Data);
        }
        catch (CRLException e) {
            Assert.fail((String)("Obtaining CRLs from X509Data failed: " + e));
        }
        Assert.assertEquals((int)javaCRLList.size(), (int)this.numExpectedCRLs, (String)"# of CRLs returned");
        Assert.assertEquals((String)((java.security.cert.X509CRL)javaCRLList.get(0)).getIssuerX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued CRL Signing Key CA,O=Test Certificates,C=US", (String)"CRL IssuerDN");
        try {
            javaCRLList = KeyInfoSupport.getCRLs((KeyInfo)this.keyInfo);
        }
        catch (CRLException e) {
            Assert.fail((String)("Obtaining CRLs from KeInfo failed: " + e));
        }
        Assert.assertEquals((int)javaCRLList.size(), (int)this.numExpectedCRLs, (String)"# of CRLs returned");
        Assert.assertEquals((String)((java.security.cert.X509CRL)javaCRLList.get(0)).getIssuerX500Principal().getName("RFC2253"), (String)"CN=Basic Self-Issued CRL Signing Key CA,O=Test Certificates,C=US", (String)"CRL IssuerDN");
    }

    @Test
    public void testCertConversionJavaToXML() throws CertificateException {
        X509Certificate xmlCert = null;
        try {
            xmlCert = KeyInfoSupport.buildX509Certificate((java.security.cert.X509Certificate)this.javaCert1);
        }
        catch (CertificateEncodingException e) {
            Assert.fail((String)("Conversion from Java X509Certificate to XMLObject failed: " + e));
        }
        Assert.assertEquals((Object)X509Support.decodeCertificate((String)xmlCert.getValue()), (Object)this.javaCert1, (String)"Java X509Certificate encoding to XMLObject failed");
    }

    @Test
    public void testCRLConversionJavaToXML() throws CertificateException, CRLException {
        X509CRL xmlCRL = null;
        try {
            xmlCRL = KeyInfoSupport.buildX509CRL((java.security.cert.X509CRL)this.javaCRL1);
        }
        catch (CRLException e) {
            Assert.fail((String)("Conversion from Java X509CRL to XMLObject failed: " + e));
        }
        Assert.assertEquals((Object)X509Support.decodeCRL((String)xmlCRL.getValue()), (Object)this.javaCRL1, (String)"Java X509CRL encoding to XMLObject failed");
    }

    @Test
    public void testDSAConversionXMLToJava() {
        PublicKey key = null;
        DSAPublicKey dsaKey = null;
        try {
            key = KeyInfoSupport.getDSAKey((DSAKeyValue)this.xmlDSAKeyValue1);
        }
        catch (KeyException e) {
            Assert.fail((String)("DSA key conversion XML to Java failed: " + e));
        }
        dsaKey = (DSAPublicKey)key;
        Assert.assertNotNull((Object)dsaKey, (String)"Generated key was not an instance of DSAPublicKey");
        Assert.assertEquals((Object)dsaKey, (Object)this.javaDSAPubKey1, (String)"Generated key was not the expected value");
        try {
            key = KeyInfoSupport.getDSAKey((DSAKeyValue)this.xmlDSAKeyValue1NoParams, (DSAParams)this.javaDSAParams1);
        }
        catch (KeyException e) {
            Assert.fail((String)("DSA key conversion XML to Java failed: " + e));
        }
        dsaKey = (DSAPublicKey)key;
        Assert.assertNotNull((Object)dsaKey, (String)"Generated key was not an instance of DSAPublicKey");
        Assert.assertEquals((Object)dsaKey, (Object)this.javaDSAPubKey1, (String)"Generated key was not the expected value");
        try {
            key = KeyInfoSupport.getDSAKey((DSAKeyValue)this.xmlDSAKeyValue1NoParams);
            Assert.fail((String)"DSA key conversion XML to Java failed should have thrown an exception but didn't");
        }
        catch (KeyException keyException) {
            // empty catch block
        }
    }

    @Test
    public void testRSAConversionXMLToJava() {
        PublicKey key = null;
        RSAPublicKey rsaKey = null;
        try {
            key = KeyInfoSupport.getRSAKey((RSAKeyValue)this.xmlRSAKeyValue1);
        }
        catch (KeyException e) {
            Assert.fail((String)("RSA key conversion XML to Java failed: " + e));
        }
        rsaKey = (RSAPublicKey)key;
        Assert.assertNotNull((Object)rsaKey, (String)"Generated key was not an instance of RSAPublicKey");
        Assert.assertEquals((Object)rsaKey, (Object)this.javaRSAPubKey1, (String)"Generated key was not the expected value");
    }

    @Test
    public void testDSAConversionJavaToXML() throws EncodingException {
        DSAKeyValue dsaKeyValue = KeyInfoSupport.buildDSAKeyValue((DSAPublicKey)this.javaDSAPubKey1);
        Assert.assertNotNull((Object)"Generated DSAKeyValue was null");
        Assert.assertEquals((Object)dsaKeyValue.getY().getValueBigInt(), (Object)this.javaDSAPubKey1.getY(), (String)"Generated DSAKeyValue Y component was not the expected value");
        Assert.assertEquals((Object)dsaKeyValue.getP().getValueBigInt(), (Object)this.javaDSAPubKey1.getParams().getP(), (String)"Generated DSAKeyValue P component was not the expected value");
        Assert.assertEquals((Object)dsaKeyValue.getQ().getValueBigInt(), (Object)this.javaDSAPubKey1.getParams().getQ(), (String)"Generated DSAKeyValue Q component was not the expected value");
        Assert.assertEquals((Object)dsaKeyValue.getG().getValueBigInt(), (Object)this.javaDSAPubKey1.getParams().getG(), (String)"Generated DSAKeyValue G component was not the expected value");
    }

    @Test
    public void testRSAConversionJavaToXML() throws EncodingException {
        RSAKeyValue rsaKeyValue = KeyInfoSupport.buildRSAKeyValue((RSAPublicKey)this.javaRSAPubKey1);
        Assert.assertNotNull((Object)"Generated RSAKeyValue was null");
        Assert.assertEquals((Object)rsaKeyValue.getModulus().getValueBigInt(), (Object)this.javaRSAPubKey1.getModulus(), (String)"Generated RSAKeyValue modulus component was not the expected value");
        Assert.assertEquals((Object)rsaKeyValue.getExponent().getValueBigInt(), (Object)this.javaRSAPubKey1.getPublicExponent(), (String)"Generated RSAKeyValue exponent component was not the expected value");
    }

    @Test
    public void testGetDSAKey() {
        this.keyValue.setRSAKeyValue(null);
        this.keyValue.setDSAKeyValue(this.xmlDSAKeyValue1);
        PublicKey pk = null;
        DSAPublicKey dsaKey = null;
        try {
            pk = KeyInfoSupport.getKey((KeyValue)this.keyValue);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of key from KeyValue failed: " + e));
        }
        Assert.assertTrue((boolean)(pk instanceof DSAPublicKey), (String)"Generated key was not an instance of DSAPublicKey");
        dsaKey = (DSAPublicKey)pk;
        Assert.assertEquals((Object)dsaKey, (Object)this.javaDSAPubKey1, (String)"Generated key was not the expected value");
        this.keyValue.setDSAKeyValue(null);
    }

    @Test
    public void testGetRSAKey() {
        this.keyValue.setDSAKeyValue(null);
        this.keyValue.setRSAKeyValue(this.xmlRSAKeyValue1);
        PublicKey pk = null;
        RSAPublicKey rsaKey = null;
        try {
            pk = KeyInfoSupport.getKey((KeyValue)this.keyValue);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of key from KeyValue failed: " + e));
        }
        Assert.assertTrue((boolean)(pk instanceof RSAPublicKey), (String)"Generated key was not an instance of RSAPublicKey");
        rsaKey = (RSAPublicKey)pk;
        Assert.assertEquals((Object)rsaKey, (Object)this.javaRSAPubKey1, (String)"Generated key was not the expected value");
        this.keyValue.setRSAKeyValue(null);
    }

    @Test
    public void testAddDSAPublicKey() throws EncodingException {
        this.keyInfo.getKeyValues().clear();
        KeyInfoSupport.addPublicKey((KeyInfo)this.keyInfo, (PublicKey)this.javaDSAPubKey1);
        KeyValue kv = (KeyValue)this.keyInfo.getKeyValues().get(0);
        Assert.assertNotNull((Object)kv, (String)"KeyValue was null");
        DSAKeyValue dsaKeyValue = kv.getDSAKeyValue();
        Assert.assertNotNull((Object)dsaKeyValue, (String)"DSAKeyValue was null");
        DSAPublicKey javaKey = null;
        try {
            javaKey = (DSAPublicKey)KeyInfoSupport.getDSAKey((DSAKeyValue)dsaKeyValue);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of Java key failed: " + e));
        }
        Assert.assertEquals((Object)javaKey, (Object)this.javaDSAPubKey1, (String)"Inserted DSA public key was not the expected value");
        this.keyInfo.getKeyValues().clear();
    }

    @Test
    public void testAddRSAPublicKey() throws EncodingException {
        this.keyInfo.getKeyValues().clear();
        KeyInfoSupport.addPublicKey((KeyInfo)this.keyInfo, (PublicKey)this.javaRSAPubKey1);
        KeyValue kv = (KeyValue)this.keyInfo.getKeyValues().get(0);
        Assert.assertNotNull((Object)kv, (String)"KeyValue was null");
        RSAKeyValue rsaKeyValue = kv.getRSAKeyValue();
        Assert.assertNotNull((Object)rsaKeyValue, (String)"RSAKeyValue was null");
        RSAPublicKey javaKey = null;
        try {
            javaKey = (RSAPublicKey)KeyInfoSupport.getRSAKey((RSAKeyValue)rsaKeyValue);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of Java key failed: " + e));
        }
        Assert.assertEquals((Object)javaKey, (Object)this.javaRSAPubKey1, (String)"Inserted RSA public key was not the expected value");
        this.keyInfo.getKeyValues().clear();
    }

    @Test
    public void testAddDEREncodedDSAPublicKey() {
        this.keyInfo.getXMLObjects(DEREncodedKeyValue.DEFAULT_ELEMENT_NAME).clear();
        try {
            KeyInfoSupport.addDEREncodedPublicKey((KeyInfo)this.keyInfo, (PublicKey)this.javaDSAPubKey1);
        }
        catch (NoSuchAlgorithmException e) {
            Assert.fail((String)("Unsupported key algorithm: " + e));
        }
        catch (InvalidKeySpecException e) {
            Assert.fail((String)("Unsupported key specification: " + e));
        }
        DEREncodedKeyValue kv = (DEREncodedKeyValue)this.keyInfo.getDEREncodedKeyValues().get(0);
        Assert.assertNotNull((Object)kv, (String)"DEREncodedKeyValue was null");
        DSAPublicKey javaKey = null;
        try {
            javaKey = (DSAPublicKey)KeyInfoSupport.getKey((DEREncodedKeyValue)kv);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of Java key failed: " + e));
        }
        Assert.assertEquals((Object)this.javaDSAPubKey1, (Object)javaKey, (String)"Inserted RSA public key was not the expected value");
        this.keyInfo.getDEREncodedKeyValues().clear();
    }

    @Test
    public void testAddDEREncodedRSAPublicKey() {
        this.keyInfo.getXMLObjects(DEREncodedKeyValue.DEFAULT_ELEMENT_NAME).clear();
        try {
            KeyInfoSupport.addDEREncodedPublicKey((KeyInfo)this.keyInfo, (PublicKey)this.javaRSAPubKey1);
        }
        catch (NoSuchAlgorithmException e) {
            Assert.fail((String)("Unsupported key algorithm: " + e));
        }
        catch (InvalidKeySpecException e) {
            Assert.fail((String)("Unsupported key specification: " + e));
        }
        DEREncodedKeyValue kv = (DEREncodedKeyValue)this.keyInfo.getDEREncodedKeyValues().get(0);
        Assert.assertNotNull((Object)kv, (String)"DEREncodedKeyValue was null");
        RSAPublicKey javaKey = null;
        try {
            javaKey = (RSAPublicKey)KeyInfoSupport.getKey((DEREncodedKeyValue)kv);
        }
        catch (KeyException e) {
            Assert.fail((String)("Extraction of Java key failed: " + e));
        }
        Assert.assertEquals((Object)this.javaRSAPubKey1, (Object)javaKey, (String)"Inserted RSA public key was not the expected value");
        this.keyInfo.getDEREncodedKeyValues().clear();
    }

    @Test
    public void testAddX509Certificate() throws CertificateException {
        this.keyInfo.getX509Datas().clear();
        KeyInfoSupport.addCertificate((KeyInfo)this.keyInfo, (java.security.cert.X509Certificate)this.javaCert1);
        X509Data x509Data = (X509Data)this.keyInfo.getX509Datas().get(0);
        Assert.assertNotNull((Object)x509Data, (String)"X509Data was null");
        X509Certificate x509Cert = (X509Certificate)x509Data.getX509Certificates().get(0);
        Assert.assertNotNull((Object)x509Cert, (String)"X509Certificate was null");
        java.security.cert.X509Certificate javaCert = null;
        javaCert = KeyInfoSupport.getCertificate((X509Certificate)x509Cert);
        Assert.assertEquals((Object)javaCert, (Object)this.javaCert1, (String)"Inserted X509Certificate was not the expected value");
        this.keyInfo.getX509Datas().clear();
    }

    @Test
    public void testAddX509CRL() throws CRLException {
        this.keyInfo.getX509Datas().clear();
        KeyInfoSupport.addCRL((KeyInfo)this.keyInfo, (java.security.cert.X509CRL)this.javaCRL1);
        X509Data x509Data = (X509Data)this.keyInfo.getX509Datas().get(0);
        Assert.assertNotNull((Object)x509Data, (String)"X509Data was null");
        X509CRL x509CRL = (X509CRL)x509Data.getX509CRLs().get(0);
        Assert.assertNotNull((Object)x509CRL, (String)"X509CRL was null");
        java.security.cert.X509CRL javaCRL = null;
        javaCRL = KeyInfoSupport.getCRL((X509CRL)x509CRL);
        Assert.assertEquals((Object)javaCRL, (Object)this.javaCRL1, (String)"Inserted X509CRL was not the expected value");
        this.keyInfo.getX509Datas().clear();
    }

    @Test
    public void testBuildSubjectName() {
        String name = "cn=foobar.example.org, o=Internet2";
        X509SubjectName xmlSubjectName = KeyInfoSupport.buildX509SubjectName((String)name);
        Assert.assertNotNull((Object)xmlSubjectName, (String)"Constructed X509SubjectName was null");
        Assert.assertEquals((String)xmlSubjectName.getValue(), (String)name, (String)"Unexpected subject name value");
    }

    @Test
    public void testBuildIssuerSerial() {
        String name = "cn=CA.example.org, o=Internet2";
        BigInteger serialNumber = new BigInteger("42");
        X509IssuerSerial xmlIssuerSerial = KeyInfoSupport.buildX509IssuerSerial((String)name, (BigInteger)serialNumber);
        Assert.assertNotNull((Object)xmlIssuerSerial, (String)"Constructed X509IssuerSerial was null");
        Assert.assertNotNull((Object)xmlIssuerSerial.getX509IssuerName(), (String)"Constructed X509IssuerName was null");
        Assert.assertEquals((String)xmlIssuerSerial.getX509IssuerName().getValue(), (String)name, (String)"Unexpected issuer name value");
        Assert.assertNotNull((Object)xmlIssuerSerial.getX509SerialNumber(), (String)"Constructed X509SerialNumber was null");
        Assert.assertEquals((Object)xmlIssuerSerial.getX509SerialNumber().getValue(), (Object)serialNumber, (String)"Unexpected serial number");
    }

    @Test
    public void testBuildSubjectKeyIdentifier() throws CertificateException, DecodingException, SecurityException {
        byte[] skiValue = Base64Support.decode((String)"r7n5HcJFGMy4IeKnR7xJvRm1eCg=");
        X509SKI xmlSKI = KeyInfoSupport.buildX509SKI((java.security.cert.X509Certificate)this.javaCert1);
        Assert.assertNotNull((Object)xmlSKI, (String)"Constructed X509SKI was null");
        Assert.assertFalse((boolean)Strings.isNullOrEmpty((String)xmlSKI.getValue()), (String)"SKI value was empty");
        byte[] xmlValue = Base64Support.decode((String)xmlSKI.getValue());
        Assert.assertNotNull((Object)xmlValue, (String)"Decoded XML SKI value was null");
        Assert.assertTrue((boolean)Arrays.equals(skiValue, xmlValue), (String)"Incorrect SKI value");
        java.security.cert.X509Certificate noExtCert = X509Support.decodeCertificate((String)"MIIBwjCCASugAwIBAgIJAMrW6QSeKNBJMA0GCSqGSIb3DQEBBAUAMCMxITAfBgNVBAMTGG5vZXh0ZW5zaW9ucy5leGFtcGxlLm9yZzAeFw0wNzA1MTkxNzU2NTVaFw0wNzA2MTgxNzU2NTVaMCMxITAfBgNVBAMTGG5vZXh0ZW5zaW9ucy5leGFtcGxlLm9yZzCBnzANBgkqhkiG9w0BAQEFAAOBjQAwgYkCgYEAw8xxu6TLqEdmnyXVZjiUoRHN6yHyobZaRK+tBEmWkD4nTlOVmTKWBCO/F4OnugaJbSTH+7Jk37l8/XYYBSIkW0+L2BglzQ2JCux/uoRu146QDIk9f5PIFs+Fxy7VRVUUZiOsonB/PNVqA7OVbPxzr1SKPSE0s9CHaDjCaEs2BnMCAwEAATANBgkqhkiG9w0BAQQFAAOBgQAuI/l80wb8K6RT1EKrAcfr9JAlJR4jmVnCK7j3Ulx++U98ze2G6/cluLxrbnqwXmxJNC3nt6xkQVJUX1UFg+zkmRrst2Nv8TTrR7S30az068BHfrZLRSUConG9jXXj+hJq+w/ojmrq8MzvJSczkA2BvsEUBARYo53na7RMgk+xWg==");
        Assert.assertNotNull((Object)noExtCert);
        X509SKI noExtXMLSKI = KeyInfoSupport.buildX509SKI((java.security.cert.X509Certificate)noExtCert);
        Assert.assertNull((Object)noExtXMLSKI, (String)"Building X509SKI from cert without SKI should have generated null");
    }

    @Test
    public void testBuildDigest() throws CertificateException, DecodingException {
        byte[] digestValue = Base64Support.decode((String)"EmkP8ttMw28A/JoA3KcO11eez7Q=");
        X509Digest xmlDigest = null;
        try {
            xmlDigest = KeyInfoSupport.buildX509Digest((java.security.cert.X509Certificate)this.javaCert1, (String)"http://www.w3.org/2000/09/xmldsig#sha1");
        }
        catch (NoSuchAlgorithmException e) {
            Assert.fail((String)("Digest algorithm missing: " + e));
        }
        Assert.assertNotNull((Object)xmlDigest, (String)"Constructed X509Digest was null");
        Assert.assertFalse((boolean)Strings.isNullOrEmpty((String)xmlDigest.getValue()), (String)"Digest value was empty");
        byte[] xmlValue = Base64Support.decode((String)xmlDigest.getValue());
        Assert.assertNotNull((Object)xmlValue, (String)"Decoded X509Digest value was null");
        Assert.assertTrue((boolean)Arrays.equals(digestValue, xmlValue), (String)"Incorrect digest value");
    }
}

