/*
 * Decompiled with CFR 0.152.
 */
package io.smallrye.jwt.build.impl;

import io.smallrye.jwt.algorithm.ContentEncryptionAlgorithm;
import io.smallrye.jwt.algorithm.KeyEncryptionAlgorithm;
import io.smallrye.jwt.build.JwtEncryptionBuilder;
import io.smallrye.jwt.build.JwtEncryptionException;
import io.smallrye.jwt.build.impl.ImplMessages;
import io.smallrye.jwt.build.impl.JwtBuildUtils;
import io.smallrye.jwt.util.KeyUtils;
import io.smallrye.jwt.util.ResourceUtils;
import java.io.InputStream;
import java.security.Key;
import java.security.PublicKey;
import java.security.interfaces.ECPublicKey;
import java.security.interfaces.RSAPublicKey;
import java.util.HashMap;
import java.util.Map;
import javax.crypto.SecretKey;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwe.JsonWebEncryption;
import org.jose4j.jwk.JsonWebKey;
import org.jose4j.lang.JoseException;

class JwtEncryptionImpl
implements JwtEncryptionBuilder {
    private static final String XEC_PUBLIC_KEY_INTERFACE = "java.security.interfaces.XECPublicKey";
    boolean innerSigned;
    String claims;
    Map<String, Object> headers = new HashMap<String, Object>();

    JwtEncryptionImpl(String claims) {
        this.claims = claims;
    }

    JwtEncryptionImpl(String claims, boolean innerSigned) {
        this.claims = claims;
        this.innerSigned = innerSigned;
    }

    @Override
    public String encrypt(PublicKey keyEncryptionKey) throws JwtEncryptionException {
        return this.encryptInternal(keyEncryptionKey);
    }

    @Override
    public String encrypt(SecretKey keyEncryptionKey) throws JwtEncryptionException {
        return this.encryptInternal(keyEncryptionKey);
    }

    @Override
    public String encrypt(String keyLocation) throws JwtEncryptionException {
        try {
            return this.encryptInternal(this.getEncryptionKeyFromKeyContent(JwtEncryptionImpl.getKeyContentFromLocation(keyLocation)));
        }
        catch (JwtEncryptionException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw ImplMessages.msg.encryptionException(ex);
        }
    }

    @Override
    public String encrypt() throws JwtEncryptionException {
        try {
            Key key = null;
            String keyLocation = JwtBuildUtils.getConfigProperty("smallrye.jwt.encrypt.key.location", String.class);
            if (keyLocation != null) {
                InputStream is;
                key = JwtBuildUtils.readPublicKeyFromKeystore(keyLocation.trim());
                if (key == null && (is = ResourceUtils.getResourceStream(keyLocation.trim())) != null) {
                    try (InputStream keyStream = is;){
                        key = this.getEncryptionKeyFromKeyContent(new String(ResourceUtils.readBytes(keyStream)));
                    }
                }
            } else {
                key = JwtBuildUtils.readPublicKeyFromKeystore(null);
                if (key == null) {
                    String keyContent = JwtBuildUtils.getConfigProperty("smallrye.jwt.encrypt.key", String.class);
                    if (keyContent != null) {
                        key = this.getEncryptionKeyFromKeyContent(keyContent);
                    } else {
                        throw ImplMessages.msg.encryptionKeyNotConfigured();
                    }
                }
            }
            if (key == null) {
                throw ImplMessages.msg.encryptionKeyCanNotBeCreatedFromContent();
            }
            return this.encryptInternal(key);
        }
        catch (JwtEncryptionException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw ImplMessages.msg.encryptionException(ex);
        }
    }

    @Override
    public String encryptWithSecret(String secret) throws JwtEncryptionException {
        return this.encrypt(KeyUtils.createSecretKeyFromSecret(secret));
    }

    @Override
    public JwtEncryptionBuilder header(String name, Object value) {
        if ("alg".equals(name)) {
            return this.keyAlgorithm(JwtEncryptionImpl.toKeyEncryptionAlgorithm((String)value));
        }
        if ("enc".equals(name)) {
            return this.contentAlgorithm(JwtEncryptionImpl.toContentEncryptionAlgorithm((String)value));
        }
        this.headers.put(name, value);
        return this;
    }

    @Override
    public JwtEncryptionBuilder keyAlgorithm(KeyEncryptionAlgorithm algorithm) {
        this.headers.put("alg", algorithm.getAlgorithm());
        return this;
    }

    @Override
    public JwtEncryptionBuilder contentAlgorithm(ContentEncryptionAlgorithm algorithm) {
        this.headers.put("enc", algorithm.getAlgorithm());
        return this;
    }

    @Override
    public JwtEncryptionBuilder keyId(String keyId) {
        this.headers.put("kid", keyId);
        return this;
    }

    private String encryptInternal(Key key) {
        if (key == null) {
            throw ImplMessages.msg.encryptionKeyIsNull();
        }
        JsonWebEncryption jwe = new JsonWebEncryption();
        jwe.setPlaintext(this.claims);
        for (Map.Entry<String, Object> entry : this.headers.entrySet()) {
            jwe.getHeaders().setObjectHeaderValue(entry.getKey(), entry.getValue());
        }
        if (this.innerSigned && !this.headers.containsKey("cty")) {
            jwe.getHeaders().setObjectHeaderValue("cty", "JWT");
        }
        String keyAlgorithm = this.getKeyEncryptionAlgorithm(key);
        jwe.setAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, keyAlgorithm));
        jwe.setAlgorithmHeaderValue(keyAlgorithm);
        jwe.setEncryptionMethodHeaderParameter(this.getContentEncryptionAlgorithm());
        jwe.setKey(key);
        if (this.isRelaxKeyValidation()) {
            jwe.setDoKeyValidation(false);
        }
        try {
            return jwe.getCompactSerialization();
        }
        catch (JoseException ex) {
            throw ImplMessages.msg.joseSerializationError(ex.getMessage(), ex);
        }
    }

    private boolean isRelaxKeyValidation() {
        return JwtBuildUtils.getConfigProperty("smallrye.jwt.encrypt.relax-key-validation", Boolean.class, false);
    }

    private String getConfiguredKeyEncryptionAlgorithm() {
        String alg = (String)this.headers.get("alg");
        if (alg == null) {
            try {
                alg = JwtBuildUtils.getConfigProperty("smallrye.jwt.new-token.key-encryption-algorithm", String.class);
                if (alg != null) {
                    alg = KeyEncryptionAlgorithm.fromAlgorithm(alg).getAlgorithm();
                    this.headers.put("alg", alg);
                }
            }
            catch (Exception ex) {
                throw ImplMessages.msg.unsupportedKeyEncryptionAlgorithm(alg);
            }
        }
        return alg;
    }

    private String getKeyEncryptionAlgorithm(Key keyEncryptionKey) {
        String alg = this.getConfiguredKeyEncryptionAlgorithm();
        if (keyEncryptionKey instanceof RSAPublicKey) {
            if (alg == null) {
                return KeyEncryptionAlgorithm.RSA_OAEP.getAlgorithm();
            }
            if (alg.startsWith("RS")) {
                return alg;
            }
        } else if (keyEncryptionKey instanceof ECPublicKey || JwtEncryptionImpl.isXecPublicKey(keyEncryptionKey)) {
            if (alg == null) {
                return KeyEncryptionAlgorithm.ECDH_ES_A256KW.getAlgorithm();
            }
            if (alg.startsWith("EC")) {
                return alg;
            }
        } else if (keyEncryptionKey instanceof SecretKey) {
            if (alg == null) {
                return KeyEncryptionAlgorithm.A256KW.getAlgorithm();
            }
            if (alg.startsWith("A") || alg.startsWith("PBE") || KeyEncryptionAlgorithm.DIR.getAlgorithm().equals(alg)) {
                return alg;
            }
        }
        throw ImplMessages.msg.unsupportedKeyEncryptionAlgorithm(keyEncryptionKey.getAlgorithm());
    }

    private static boolean isXecPublicKey(Key encKey) {
        return KeyUtils.isSupportedKey(encKey, XEC_PUBLIC_KEY_INTERFACE);
    }

    private String getContentEncryptionAlgorithm() {
        String alg = (String)this.headers.get("enc");
        if (alg == null) {
            try {
                alg = JwtBuildUtils.getConfigProperty("smallrye.jwt.new-token.content-encryption-algorithm", String.class);
                if (alg != null) {
                    alg = ContentEncryptionAlgorithm.fromAlgorithm(alg).getAlgorithm();
                }
            }
            catch (Exception ex) {
                throw ImplMessages.msg.unsupportedContentEncryptionAlgorithm(alg);
            }
        }
        return alg != null ? alg : ContentEncryptionAlgorithm.A256GCM.name();
    }

    private static String getKeyContentFromLocation(String keyLocation) {
        try {
            return KeyUtils.readKeyContent(keyLocation);
        }
        catch (Exception ex) {
            throw ImplMessages.msg.encryptionKeyCanNotBeLoadedFromLocation(keyLocation);
        }
    }

    Key getEncryptionKeyFromKeyContent(String keyContent) {
        String kid = (String)this.headers.get("kid");
        String alg = this.getConfiguredKeyEncryptionAlgorithm();
        Key key = KeyUtils.tryAsPemEncryptionPublicKey(keyContent, alg == null ? KeyEncryptionAlgorithm.RSA_OAEP_256 : KeyEncryptionAlgorithm.fromAlgorithm(alg));
        if (key == null) {
            JsonWebKey jwk;
            if (kid == null && (kid = JwtBuildUtils.getConfigProperty("smallrye.jwt.encrypt.key.id", String.class)) != null) {
                this.headers.put("kid", kid);
            }
            if ((jwk = KeyUtils.getJwkKeyFromJwkSet(kid, keyContent)) != null && (key = KeyUtils.getPublicOrSecretEncryptingKey(jwk, alg == null ? null : KeyEncryptionAlgorithm.fromAlgorithm(alg))) != null) {
                if (alg == null && jwk.getAlgorithm() != null) {
                    this.headers.put("alg", jwk.getAlgorithm());
                }
                if (kid == null && jwk.getKeyId() != null) {
                    this.headers.put("kid", jwk.getKeyId());
                }
            }
        }
        if (key == null) {
            throw ImplMessages.msg.encryptionKeyCanNotBeCreatedFromContent();
        }
        return key;
    }

    private static KeyEncryptionAlgorithm toKeyEncryptionAlgorithm(String value) {
        try {
            return KeyEncryptionAlgorithm.fromAlgorithm(value);
        }
        catch (Exception ex) {
            throw ImplMessages.msg.unsupportedKeyEncryptionAlgorithm(value);
        }
    }

    private static ContentEncryptionAlgorithm toContentEncryptionAlgorithm(String value) {
        try {
            return ContentEncryptionAlgorithm.fromAlgorithm(value);
        }
        catch (Exception ex) {
            throw ImplMessages.msg.unsupportedContentEncryptionAlgorithm(value);
        }
    }
}

