/*
 * Decompiled with CFR 0.152.
 */
package com.orientechnologies.orient.core.security;

import com.orientechnologies.common.collection.OLRUCache;
import com.orientechnologies.common.exception.OException;
import com.orientechnologies.common.log.OLogManager;
import com.orientechnologies.orient.core.config.OGlobalConfiguration;
import com.orientechnologies.orient.core.exception.OConfigurationException;
import com.orientechnologies.orient.core.exception.OSecurityException;
import com.orientechnologies.orient.core.metadata.security.OSecurity;
import com.orientechnologies.orient.core.security.OCredentialInterceptor;
import com.orientechnologies.orient.core.security.OSecurityFactory;
import com.orientechnologies.orient.core.security.OSecuritySharedFactory;
import java.io.UnsupportedEncodingException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;

public class OSecurityManager {
    public static final String HASH_ALGORITHM = "SHA-256";
    public static final String HASH_ALGORITHM_PREFIX = "{SHA-256}";
    public static final String PBKDF2_ALGORITHM = "PBKDF2WithHmacSHA1";
    public static final String PBKDF2_ALGORITHM_PREFIX = "{PBKDF2WithHmacSHA1}";
    public static final String PBKDF2_SHA256_ALGORITHM = "PBKDF2WithHmacSHA256";
    public static final String PBKDF2_SHA256_ALGORITHM_PREFIX = "{PBKDF2WithHmacSHA256}";
    public static final int SALT_SIZE = 24;
    public static final int HASH_SIZE = 24;
    private static final OSecurityManager instance = new OSecurityManager();
    private volatile OSecurityFactory securityFactory = new OSecuritySharedFactory();
    private MessageDigest md;
    private static Map<String, byte[]> SALT_CACHE = null;

    public OSecurityManager() {
        try {
            this.md = MessageDigest.getInstance(HASH_ALGORITHM);
        }
        catch (NoSuchAlgorithmException e) {
            OLogManager.instance().error((Object)this, "Cannot use OSecurityManager", e, new Object[0]);
        }
    }

    public static String createHash(String iInput, String iAlgorithm) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        if (iAlgorithm == null) {
            iAlgorithm = HASH_ALGORITHM;
        }
        MessageDigest msgDigest = MessageDigest.getInstance(iAlgorithm);
        return OSecurityManager.byteArrayToHexStr(msgDigest.digest(iInput.getBytes("UTF-8")));
    }

    public static OSecurityManager instance() {
        return instance;
    }

    public boolean checkPassword(String iPassword, String iHash) {
        if (iHash.startsWith(HASH_ALGORITHM_PREFIX)) {
            String s = iHash.substring(HASH_ALGORITHM_PREFIX.length());
            return this.createSHA256(iPassword).equals(s);
        }
        if (iHash.startsWith(PBKDF2_ALGORITHM_PREFIX)) {
            String s = iHash.substring(PBKDF2_ALGORITHM_PREFIX.length());
            return this.checkPasswordWithSalt(iPassword, s, PBKDF2_ALGORITHM);
        }
        if (iHash.startsWith(PBKDF2_SHA256_ALGORITHM_PREFIX)) {
            String s = iHash.substring(PBKDF2_SHA256_ALGORITHM_PREFIX.length());
            return this.checkPasswordWithSalt(iPassword, s, PBKDF2_SHA256_ALGORITHM);
        }
        return MessageDigest.isEqual(this.digestSHA256(iPassword), this.digestSHA256(iHash));
    }

    public String createSHA256(String iInput) {
        return OSecurityManager.byteArrayToHexStr(this.digestSHA256(iInput));
    }

    public String createHash(String iInput, String iAlgorithm, boolean iIncludeAlgorithm) {
        String transformed;
        if (iInput == null) {
            throw new IllegalArgumentException("Input string is null");
        }
        if (iAlgorithm == null) {
            throw new IllegalArgumentException("Algorithm is null");
        }
        StringBuilder buffer = new StringBuilder(128);
        String algorithm = this.validateAlgorithm(iAlgorithm);
        if (iIncludeAlgorithm) {
            buffer.append('{');
            buffer.append(algorithm);
            buffer.append('}');
        }
        if (HASH_ALGORITHM.equalsIgnoreCase(algorithm)) {
            transformed = this.createSHA256(iInput);
        } else if (PBKDF2_ALGORITHM.equalsIgnoreCase(algorithm)) {
            transformed = this.createHashWithSalt(iInput, OGlobalConfiguration.SECURITY_USER_PASSWORD_SALT_ITERATIONS.getValueAsInteger(), algorithm);
        } else if (PBKDF2_SHA256_ALGORITHM.equalsIgnoreCase(algorithm)) {
            transformed = this.createHashWithSalt(iInput, OGlobalConfiguration.SECURITY_USER_PASSWORD_SALT_ITERATIONS.getValueAsInteger(), algorithm);
        } else {
            throw new IllegalArgumentException("Algorithm '" + algorithm + "' is not supported");
        }
        buffer.append(transformed);
        return buffer.toString();
    }

    public synchronized byte[] digestSHA256(String iInput) {
        if (iInput == null) {
            return null;
        }
        try {
            return this.md.digest(iInput.getBytes("UTF-8"));
        }
        catch (UnsupportedEncodingException e) {
            String message = "The requested encoding is not supported: cannot execute security checks";
            OLogManager.instance().error((Object)this, "The requested encoding is not supported: cannot execute security checks", e, new Object[0]);
            throw OException.wrapException(new OConfigurationException("The requested encoding is not supported: cannot execute security checks"), e);
        }
    }

    public String createHashWithSalt(String iPassword) {
        return this.createHashWithSalt(iPassword, OGlobalConfiguration.SECURITY_USER_PASSWORD_SALT_ITERATIONS.getValueAsInteger(), OGlobalConfiguration.SECURITY_USER_PASSWORD_DEFAULT_ALGORITHM.getValueAsString());
    }

    public String createHashWithSalt(String iPassword, int iIterations, String algorithm) {
        SecureRandom random = new SecureRandom();
        byte[] salt = new byte[24];
        random.nextBytes(salt);
        byte[] hash = this.getPbkdf2(iPassword, salt, iIterations, 24, this.validateAlgorithm(algorithm));
        return OSecurityManager.byteArrayToHexStr(hash) + ":" + OSecurityManager.byteArrayToHexStr(salt) + ":" + iIterations;
    }

    public boolean checkPasswordWithSalt(String iPassword, String iHash) {
        return this.checkPasswordWithSalt(iPassword, iHash, OGlobalConfiguration.SECURITY_USER_PASSWORD_DEFAULT_ALGORITHM.getValueAsString());
    }

    public boolean checkPasswordWithSalt(String iPassword, String iHash, String algorithm) {
        if (!OSecurityManager.isAlgorithmSupported(algorithm)) {
            OLogManager.instance().error((Object)this, "The password hash algorithm is not supported: %s", algorithm);
            return false;
        }
        String[] params = iHash.split(":");
        if (params.length != 3) {
            throw new IllegalArgumentException("Hash does not contain the requested parts: <hash>:<salt>:<iterations>");
        }
        byte[] hash = OSecurityManager.hexToByteArray(params[0]);
        byte[] salt = OSecurityManager.hexToByteArray(params[1]);
        int iterations = Integer.parseInt(params[2]);
        byte[] testHash = this.getPbkdf2(iPassword, salt, iterations, hash.length, algorithm);
        return MessageDigest.isEqual(hash, testHash);
    }

    private byte[] getPbkdf2(String iPassword, byte[] salt, int iterations, int bytes, String algorithm) {
        byte[] encoded;
        String cacheKey = null;
        String hashedPassword = this.createSHA256(iPassword + new String(salt));
        if (SALT_CACHE != null && (encoded = SALT_CACHE.get(cacheKey = hashedPassword + "|" + Arrays.toString(salt) + "|" + iterations + "|" + bytes)) != null) {
            return encoded;
        }
        PBEKeySpec spec = new PBEKeySpec(iPassword.toCharArray(), salt, iterations, bytes * 8);
        try {
            SecretKeyFactory skf = SecretKeyFactory.getInstance(algorithm);
            byte[] encoded2 = skf.generateSecret(spec).getEncoded();
            if (SALT_CACHE != null) {
                SALT_CACHE.put(cacheKey, encoded2);
            }
            return encoded2;
        }
        catch (Exception e) {
            throw OException.wrapException(new OSecurityException("Cannot create a key with '" + algorithm + "' algorithm"), e);
        }
    }

    private static boolean isAlgorithmSupported(String algorithm) {
        return !Runtime.class.getPackage().getImplementationVersion().startsWith("1.7") || !algorithm.equals(PBKDF2_SHA256_ALGORITHM);
    }

    private String validateAlgorithm(String iAlgorithm) {
        String validAlgo = iAlgorithm;
        if (!OSecurityManager.isAlgorithmSupported(iAlgorithm)) {
            validAlgo = PBKDF2_ALGORITHM;
            OLogManager.instance().debug((Object)this, "The %s algorithm is not supported, downgrading to %s", iAlgorithm, validAlgo);
        }
        return validAlgo;
    }

    public static String byteArrayToHexStr(byte[] data) {
        if (data == null) {
            return null;
        }
        char[] chars = new char[data.length * 2];
        for (int i = 0; i < data.length; ++i) {
            byte current = data[i];
            int hi = (current & 0xF0) >> 4;
            int lo = current & 0xF;
            chars[2 * i] = (char)(hi < 10 ? 48 + hi : 65 + hi - 10);
            chars[2 * i + 1] = (char)(lo < 10 ? 48 + lo : 65 + lo - 10);
        }
        return new String(chars);
    }

    private static byte[] hexToByteArray(String data) {
        byte[] hex = new byte[data.length() / 2];
        for (int i = 0; i < hex.length; ++i) {
            hex[i] = (byte)Integer.parseInt(data.substring(2 * i, 2 * i + 2), 16);
        }
        return hex;
    }

    public OCredentialInterceptor newCredentialInterceptor() {
        OCredentialInterceptor ci = null;
        try {
            Class<?> cls;
            String ciClass = OGlobalConfiguration.CLIENT_CREDENTIAL_INTERCEPTOR.getValueAsString();
            if (ciClass != null && OCredentialInterceptor.class.isAssignableFrom(cls = Class.forName(ciClass))) {
                ci = (OCredentialInterceptor)cls.newInstance();
            }
        }
        catch (Exception ex) {
            OLogManager.instance().debug((Object)this, "newCredentialInterceptor() Exception creating CredentialInterceptor", ex, new Object[0]);
        }
        return ci;
    }

    public OSecurityFactory getSecurityFactory() {
        return this.securityFactory;
    }

    public void setSecurityFactory(OSecurityFactory factory) {
        this.securityFactory = factory != null ? factory : new OSecuritySharedFactory();
    }

    public OSecurity newSecurity() {
        if (this.securityFactory != null) {
            return this.securityFactory.newSecurity();
        }
        return null;
    }

    static {
        int cacheSize = OGlobalConfiguration.SECURITY_USER_PASSWORD_SALT_CACHE_SIZE.getValueAsInteger();
        if (cacheSize > 0) {
            SALT_CACHE = Collections.synchronizedMap(new OLRUCache(cacheSize));
        }
    }
}

