/*
 * Decompiled with CFR 0.152.
 */
package org.apache.activemq.artemis.core.journal.impl;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.BlockingDeque;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Stream;
import org.apache.activemq.artemis.api.core.ActiveMQIOErrorException;
import org.apache.activemq.artemis.core.io.SequentialFile;
import org.apache.activemq.artemis.core.io.SequentialFileFactory;
import org.apache.activemq.artemis.core.journal.impl.JournalFile;
import org.apache.activemq.artemis.core.journal.impl.JournalFileImpl;
import org.apache.activemq.artemis.core.journal.impl.JournalImpl;
import org.apache.activemq.artemis.journal.ActiveMQJournalBundle;
import org.apache.activemq.artemis.journal.ActiveMQJournalLogger;
import org.jboss.logging.Logger;

public class JournalFilesRepository {
    private static final Logger logger = Logger.getLogger(JournalFilesRepository.class);
    private static final boolean CHECK_CONSISTENCE = false;
    private final SequentialFileFactory fileFactory;
    private final JournalImpl journal;
    private final BlockingDeque<JournalFile> dataFiles = new LinkedBlockingDeque<JournalFile>();
    private final ConcurrentLinkedQueue<JournalFile> freeFiles = new ConcurrentLinkedQueue();
    private final BlockingQueue<JournalFile> openedFiles = new LinkedBlockingQueue<JournalFile>();
    private final AtomicLong nextFileID = new AtomicLong(0L);
    private final int maxAIO;
    private final int minFiles;
    private final int poolSize;
    private final int fileSize;
    private final String filePrefix;
    private final String fileExtension;
    private final int userVersion;
    private final AtomicInteger freeFilesCount = new AtomicInteger(0);
    private final int journalFileOpenTimeout;
    private final int maxAtticFiles;
    private Executor openFilesExecutor;
    private final Runnable pushOpenRunnable = new Runnable(){

        @Override
        public void run() {
            try {
                JournalFilesRepository.this.pushOpenedFile();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorPushingFile(e);
                JournalFilesRepository.this.fileFactory.onIOError(e, "unable to open ", null);
            }
        }
    };

    public JournalFilesRepository(SequentialFileFactory fileFactory, JournalImpl journal, String filePrefix, String fileExtension, int userVersion, int maxAIO, int fileSize, int minFiles, int poolSize, int journalFileOpenTimeout, int maxAtticFiles) {
        if (filePrefix == null) {
            throw new IllegalArgumentException("filePrefix cannot be null");
        }
        if (fileExtension == null) {
            throw new IllegalArgumentException("fileExtension cannot be null");
        }
        if (maxAIO <= 0) {
            throw new IllegalArgumentException("maxAIO must be a positive number");
        }
        this.fileFactory = fileFactory;
        this.maxAIO = maxAIO;
        this.filePrefix = filePrefix;
        this.fileExtension = fileExtension;
        this.minFiles = minFiles;
        this.fileSize = fileSize;
        this.poolSize = poolSize;
        this.userVersion = userVersion;
        this.journal = journal;
        this.journalFileOpenTimeout = journalFileOpenTimeout;
        this.maxAtticFiles = maxAtticFiles;
    }

    public int getPoolSize() {
        return this.poolSize;
    }

    public void setExecutor(Executor fileExecutor) {
        this.openFilesExecutor = fileExecutor;
    }

    public void clear() throws Exception {
        for (JournalFile file : this.dataFiles) {
            file.getFile().waitNotPending();
        }
        this.dataFiles.clear();
        for (JournalFile file : this.freeFiles) {
            file.getFile().waitNotPending();
        }
        this.freeFiles.clear();
        this.freeFilesCount.set(0);
        for (JournalFile file : this.openedFiles) {
            try {
                file.getFile().close();
            }
            catch (Exception e) {
                ActiveMQJournalLogger.LOGGER.errorClosingFile(e);
            }
        }
        this.openedFiles.clear();
    }

    public int getMaxAIO() {
        return this.maxAIO;
    }

    public String getFileExtension() {
        return this.fileExtension;
    }

    public String getFilePrefix() {
        return this.filePrefix;
    }

    public void calculateNextfileID(List<JournalFile> files) {
        for (JournalFile file : files) {
            long fileIdFromFile = file.getFileID();
            long fileIdFromName = JournalFilesRepository.getFileNameID(this.filePrefix, file.getFile().getFileName());
            this.setNextFileID(Math.max(fileIdFromName, fileIdFromFile));
        }
    }

    public void setNextFileID(long targetUpdate) {
        long current;
        do {
            if ((current = this.nextFileID.get()) < targetUpdate) continue;
            return;
        } while (!this.nextFileID.compareAndSet(current, targetUpdate));
    }

    public void ensureMinFiles() throws Exception {
        int filesToCreate = this.minFiles - (this.dataFiles.size() + this.freeFilesCount.get());
        if (filesToCreate > 0) {
            for (int i = 0; i < filesToCreate; ++i) {
                this.freeFiles.add(this.createFile(false, false, true, false, -1L));
                this.freeFilesCount.getAndIncrement();
            }
        }
    }

    public void openFile(JournalFile file, boolean multiAIO) throws Exception {
        if (multiAIO) {
            file.getFile().open();
        } else {
            file.getFile().open(1, false);
        }
        file.getFile().position(file.getFile().calculateBlockStart(16));
    }

    public JournalFile[] getDataFilesArray() {
        return this.dataFiles.toArray(new JournalFile[this.dataFiles.size()]);
    }

    public JournalFile pollLastDataFile() {
        return (JournalFile)this.dataFiles.pollLast();
    }

    public void removeDataFile(JournalFile file) {
        if (!this.dataFiles.remove(file)) {
            ActiveMQJournalLogger.LOGGER.couldNotRemoveFile(file);
        }
    }

    public int getDataFilesCount() {
        return this.dataFiles.size();
    }

    public int getJournalFileOpenTimeout() {
        return this.journalFileOpenTimeout;
    }

    public Collection<JournalFile> getDataFiles() {
        return this.dataFiles;
    }

    public void clearDataFiles() {
        this.dataFiles.clear();
    }

    public void addDataFileOnTop(JournalFile file) {
        this.dataFiles.addFirst(file);
    }

    public String debugFiles() {
        StringBuilder buffer = new StringBuilder();
        buffer.append("**********\nCurrent File = " + this.journal.getCurrentFile() + "\n");
        buffer.append("**********\nDataFiles:\n");
        for (JournalFile file : this.dataFiles) {
            buffer.append(file.toString() + "\n");
        }
        buffer.append("*********\nFreeFiles:\n");
        for (JournalFile file : this.freeFiles) {
            buffer.append(file.toString() + "\n");
        }
        return buffer.toString();
    }

    public synchronized void checkDataFiles() {
        long seq = -1L;
        for (JournalFile file : this.dataFiles) {
            if (file.getFileID() <= seq) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.seqOutOfOrder();
                throw new IllegalStateException("Sequence out of order");
            }
            if (this.journal.getCurrentFile() != null && this.journal.getCurrentFile().getFileID() <= file.getFileID()) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.currentFile(file.getFileID(), this.journal.getCurrentFile().getFileID(), file.getFileID(), this.journal.getCurrentFile() == file);
            }
            if (this.journal.getCurrentFile() == file) {
                throw new RuntimeException("Check failure! Current file listed as data file!");
            }
            seq = file.getFileID();
        }
        long lastFreeId = -1L;
        for (JournalFile file : this.freeFiles) {
            if (file.getFileID() <= lastFreeId) {
                ActiveMQJournalLogger.LOGGER.checkFiles();
                ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
                ActiveMQJournalLogger.LOGGER.fileIdOutOfOrder();
                throw new RuntimeException("Check failure!");
            }
            lastFreeId = file.getFileID();
            if (file.getFileID() >= seq) continue;
            ActiveMQJournalLogger.LOGGER.checkFiles();
            ActiveMQJournalLogger.LOGGER.info(this.debugFiles());
            ActiveMQJournalLogger.LOGGER.fileTooSmall();
        }
    }

    public void addDataFileOnBottom(JournalFile file) {
        this.dataFiles.add(file);
    }

    public int getFreeFilesCount() {
        return this.freeFilesCount.get();
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp) throws Exception {
        this.addFreeFile(file, renameTmp, true);
    }

    public synchronized void addFreeFile(JournalFile file, boolean renameTmp, boolean checkDelete) throws Exception {
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("Adding free file " + file + ", renameTMP=" + renameTmp + ", checkDelete=" + checkDelete));
        }
        long calculatedSize = 0L;
        try {
            calculatedSize = file.getFile().size();
        }
        catch (Exception e) {
            throw new IllegalStateException(e.getMessage() + " file: " + file);
        }
        if (calculatedSize != (long)this.fileSize) {
            this.damagedFile(file);
        } else if (!checkDelete || this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size() < this.poolSize || this.poolSize < 0) {
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("Re-initializing file " + file + " as checkDelete=" + checkDelete + ", freeFilesCount=" + this.freeFilesCount + ", dataFiles.size=" + this.dataFiles.size() + ", openedFiles=" + this.openedFiles + ", poolSize=" + this.poolSize));
            }
            JournalFile jf = this.reinitializeFile(file);
            if (renameTmp) {
                jf.getFile().renameTo(JournalImpl.renameExtensionFile(jf.getFile().getFileName(), ".tmp"));
            }
            this.freeFiles.add(jf);
            this.freeFilesCount.getAndIncrement();
        } else {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Deleting file " + file.getFile()));
            }
            if (logger.isTraceEnabled()) {
                logger.trace((Object)("DataFiles.size() = " + this.dataFiles.size()));
                logger.trace((Object)("openedFiles.size() = " + this.openedFiles.size()));
                logger.trace((Object)("minfiles = " + this.minFiles + ", poolSize = " + this.poolSize));
                logger.trace((Object)("Free Files = " + this.freeFilesCount.get()));
                logger.trace((Object)("File " + file + " being deleted as freeFiles.size() + dataFiles.size() + 1 + openedFiles.size() (" + (this.freeFilesCount.get() + this.dataFiles.size() + 1 + this.openedFiles.size()) + ") < minFiles (" + this.minFiles + ")"));
            }
            file.getFile().delete();
        }
    }

    private void damagedFile(JournalFile file) throws Exception {
        if (file.getFile().isOpen()) {
            file.getFile().close(false, false);
        }
        if (file.getFile().exists()) {
            Path journalPath = file.getFile().getJavaFile().toPath();
            Path atticPath = journalPath.getParent().resolve("attic");
            Files.createDirectories(atticPath, new FileAttribute[0]);
            if (this.listFiles(atticPath) < this.maxAtticFiles) {
                ActiveMQJournalLogger.LOGGER.movingFileToAttic(file.getFile().getFileName());
                Files.move(journalPath, atticPath.resolve(journalPath.getFileName()), StandardCopyOption.REPLACE_EXISTING);
            } else {
                ActiveMQJournalLogger.LOGGER.deletingFile(file);
                Files.delete(journalPath);
            }
        }
    }

    private int listFiles(Path path) throws IOException {
        try (Stream<Path> files = Files.list(path);){
            int n = files.mapToInt(e -> 1).sum();
            return n;
        }
    }

    public Collection<JournalFile> getFreeFiles() {
        return this.freeFiles;
    }

    public JournalFile getFreeFile() {
        JournalFile file = (JournalFile)this.freeFiles.remove();
        this.freeFilesCount.getAndDecrement();
        return file;
    }

    public int getOpenedFilesCount() {
        return this.openedFiles.size();
    }

    public JournalFile openFileCMP() throws Exception {
        JournalFile file = this.openFile();
        SequentialFile sequentialFile = file.getFile();
        sequentialFile.close();
        sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
        return file;
    }

    public JournalFile openFile() throws InterruptedException, ActiveMQIOErrorException {
        JournalFile nextFile;
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("enqueueOpenFile with openedFiles.size=" + this.openedFiles.size()));
        }
        if ((nextFile = (JournalFile)this.openedFiles.poll()) == null) {
            this.pushOpen();
            nextFile = this.openedFiles.poll(this.journalFileOpenTimeout, TimeUnit.SECONDS);
        } else if (this.openedFiles.isEmpty()) {
            this.pushOpen();
        }
        if (nextFile == null) {
            ActiveMQJournalLogger.LOGGER.cantOpenFileTimeout(this.journalFileOpenTimeout);
            try {
                nextFile = this.takeFile(true, true, true, false);
            }
            catch (Exception e) {
                this.fileFactory.onIOError(e, "unable to open ", null);
                this.fileFactory.activateBuffer(this.journal.getCurrentFile().getFile());
                throw ActiveMQJournalBundle.BUNDLE.fileNotOpened();
            }
        }
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Returning file " + nextFile));
        }
        return nextFile;
    }

    private void pushOpen() {
        if (this.openFilesExecutor == null) {
            this.pushOpenRunnable.run();
        } else {
            this.openFilesExecutor.execute(this.pushOpenRunnable);
        }
    }

    public synchronized void pushOpenedFile() throws Exception {
        JournalFile nextOpenedFile = this.takeFile(true, true, true, false);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("pushing openFile " + nextOpenedFile));
        }
        if (!this.openedFiles.offer(nextOpenedFile)) {
            ActiveMQJournalLogger.LOGGER.failedToAddFile(nextOpenedFile);
        }
    }

    public void closeFile(JournalFile file, boolean block) throws Exception {
        this.fileFactory.deactivateBuffer();
        file.getFile().close(true, block);
        if (!this.dataFiles.contains(file)) {
            this.dataFiles.add(file);
        }
    }

    private JournalFile takeFile(boolean keepOpened, boolean multiAIO, boolean initFile, boolean tmpCompactExtension) throws Exception {
        JournalFile nextFile = null;
        nextFile = this.freeFiles.poll();
        if (nextFile != null) {
            this.freeFilesCount.getAndDecrement();
        }
        if (nextFile == null) {
            nextFile = this.createFile(keepOpened, multiAIO, initFile, tmpCompactExtension, -1L);
        } else {
            if (tmpCompactExtension) {
                SequentialFile sequentialFile = nextFile.getFile();
                sequentialFile.renameTo(sequentialFile.getFileName() + ".cmp");
            }
            if (keepOpened) {
                this.openFile(nextFile, multiAIO);
            }
        }
        return nextFile;
    }

    public JournalFile createRemoteBackupSyncFile(long fileID) throws Exception {
        return this.createFile(false, false, true, false, fileID);
    }

    private JournalFile createFile(final boolean keepOpened, final boolean multiAIO, final boolean init, final boolean tmpCompact, final long fileIdPreSet) throws Exception {
        if (System.getSecurityManager() == null) {
            return this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet);
        }
        try {
            return AccessController.doPrivileged(new PrivilegedExceptionAction<JournalFile>(){

                @Override
                public JournalFile run() throws Exception {
                    return JournalFilesRepository.this.createFile0(keepOpened, multiAIO, init, tmpCompact, fileIdPreSet);
                }
            });
        }
        catch (PrivilegedActionException e) {
            throw this.unwrapException(e);
        }
    }

    private RuntimeException unwrapException(PrivilegedActionException e) throws Exception {
        Throwable c = e.getCause();
        if (c instanceof RuntimeException) {
            throw (RuntimeException)c;
        }
        if (c instanceof Error) {
            throw (Error)c;
        }
        throw new RuntimeException(c);
    }

    private JournalFile createFile0(boolean keepOpened, boolean multiAIO, boolean init, boolean tmpCompact, long fileIdPreSet) throws Exception {
        long fileID = fileIdPreSet != -1L ? fileIdPreSet : this.generateFileID();
        String fileName = this.createFileName(tmpCompact, fileID);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Creating file " + fileName));
        }
        String tmpFileName = fileName + ".tmp";
        SequentialFile sequentialFile = this.fileFactory.createSequentialFile(tmpFileName);
        sequentialFile.open(1, false);
        if (init) {
            sequentialFile.fill(this.fileSize);
            JournalImpl.initFileHeader(this.fileFactory, sequentialFile, this.userVersion, fileID);
        }
        long position = sequentialFile.position();
        sequentialFile.close(false, false);
        if (logger.isTraceEnabled()) {
            logger.trace((Object)("Renaming file " + tmpFileName + " as " + fileName));
        }
        sequentialFile.renameTo(fileName);
        if (keepOpened) {
            if (multiAIO) {
                sequentialFile.open();
            } else {
                sequentialFile.open(1, false);
            }
            sequentialFile.position(position);
        }
        return new JournalFileImpl(sequentialFile, fileID, 2);
    }

    private String createFileName(boolean tmpCompact, long fileID) {
        String fileName = tmpCompact ? this.filePrefix + "-" + fileID + "." + this.fileExtension + ".cmp" : this.filePrefix + "-" + fileID + "." + this.fileExtension;
        return fileName;
    }

    private long generateFileID() {
        return this.nextFileID.incrementAndGet();
    }

    public static long getFileNameID(String filePrefix, String fileName) {
        try {
            return Long.parseLong(fileName.substring(filePrefix.length() + 1, fileName.indexOf(46)));
        }
        catch (Throwable e) {
            try {
                return Long.parseLong(fileName.substring(fileName.lastIndexOf("-") + 1, fileName.indexOf(46)));
            }
            catch (Throwable e2) {
                ActiveMQJournalLogger.LOGGER.errorRetrievingID(e, fileName);
                return 0L;
            }
        }
    }

    private JournalFile reinitializeFile(JournalFile file) throws Exception {
        long newFileID = this.generateFileID();
        SequentialFile sf = file.getFile();
        sf.open(1, false);
        int position = JournalImpl.initFileHeader(this.fileFactory, sf, this.userVersion, newFileID);
        JournalFileImpl jf = new JournalFileImpl(sf, newFileID, 2);
        sf.position(position);
        sf.close(false, false);
        return jf;
    }

    public String toString() {
        return "JournalFilesRepository(dataFiles=" + this.dataFiles + ", freeFiles=" + this.freeFiles + ", openedFiles=" + this.openedFiles + ")";
    }
}

