/*
 * Copyright The WildFly Authors
 * SPDX-License-Identifier: Apache-2.0
 */

package org.jboss.as.test.smoke.jms;

import jakarta.ejb.EJB;
import jakarta.enterprise.event.Observes;
import jakarta.jms.JMSException;
import jakarta.jms.Message;
import jakarta.jms.TextMessage;

import org.jboss.arquillian.container.test.api.Deployment;
import org.jboss.arquillian.junit.Arquillian;
import org.jboss.as.arquillian.api.ServerSetup;
import org.jboss.as.test.integration.common.jms.JMSOperations;
import org.jboss.as.test.jms.auxiliary.CreateQueueSetupTask;
import org.jboss.as.test.smoke.jms.auxiliary.QueueMessageDrivenBean;
import org.jboss.as.test.smoke.jms.auxiliary.TransactedQueueMessageSender;
import org.jboss.logging.Logger;
import org.jboss.shrinkwrap.api.ShrinkWrap;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.jboss.shrinkwrap.api.spec.JavaArchive;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;

/**
 * Tests sending Jakarta Messaging messages to a queue within a transaction
 *
 * @author <a href="jmartisk@redhat.com">Jan Martiska</a>
 */
@RunWith(Arquillian.class)
@ServerSetup(CreateQueueSetupTask.class)
public class SendToQueueFromWithinTransactionTest {

    private static final Logger logger = Logger.getLogger(SendToQueueFromWithinTransactionTest.class);

    @EJB
    private TransactedQueueMessageSender sender;

    @EJB
    private TransactedQueueMessageSender sender2;

    private static volatile boolean messageReceived;

    @Before
    public void setMessageReceived() {
        messageReceived = false;
    }

    @Deployment
    public static JavaArchive createTestArchive() {
        return ShrinkWrap.create(JavaArchive.class, "test.jar")
                .addClass(TransactedQueueMessageSender.class)
                .addClass(QueueMessageDrivenBean.class)
                .addClass(CreateQueueSetupTask.class)
                .addPackage(JMSOperations.class.getPackage())
                .addAsManifestResource(new StringAsset("<beans bean-discovery-mode=\"all\"></beans>"), "beans.xml");
    }

    @Test
    public void sendSuccessfully() throws Exception {
        try {
            sender.sendToQueueSuccessfully();
            Thread.sleep(2000);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        Assert.assertTrue(messageReceived);
    }

    @Test
    public void sendAndRollback() {
        try {
            sender2.sendToQueueAndRollback();
            Thread.sleep(2000);
        } catch (Exception ex) {
            ex.printStackTrace();
        }
        Assert.assertFalse(messageReceived);
    }

    public void receivedMessage(@Observes Message message) {
        messageReceived = true;
        try {
            logger.trace("caught event... message=" + ((TextMessage) message).getText());
        } catch (JMSException ex) {
            ex.printStackTrace();
            Assert.fail(ex.getMessage());
        }
    }

}
